<?php
session_start();

error_reporting(E_ALL);
ini_set('display_errors', 1);

require '../../class/configClass.php';
require '../../vendor/autoload.php';
Config::load('../../config/config.php');
include_once('../../conexao/conexao.php');
include_once('../../funcoes/general.php');
require '../../class/pagarVendaUsuarioClass.php';
require '../../../funcoes/ProdutoImagemHelper.php';

// Inicializar helper de imagens
$imagemHelper = new ProdutoImagemHelper($pdo);

// Parâmetros de paginação
$itemsPerPage = 50; // Número de itens por página
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $itemsPerPage;

$fornecedor = $_GET['fornecedor'] ?? '';
$nomeProduto = $_GET['nomeProduto'] ?? '';
$categoriaInterna = $_GET['categoria_interna'] ?? '';

// Não aplicar filtro se o fornecedor não for válido
if ($fornecedor === 'Selecione um fornecedor...') {
    $fornecedor = '';
}

// Verificar se o fornecedor está de férias
$fornecedorEmFerias = false;
$feriasInfo = null;

if (!empty($fornecedor)) {
    $stmtFerias = $pdo->prepare("
        SELECT 
            nome,
            ferias_ativo,
            ferias_de,
            ferias_ate
        FROM fornecedores 
        WHERE nome = ? AND ativo = 1
    ");
    $stmtFerias->execute([$fornecedor]);
    $dadosFerias = $stmtFerias->fetch(PDO::FETCH_ASSOC);
    
    if ($dadosFerias && $dadosFerias['ferias_ativo'] == 1 && 
        !empty($dadosFerias['ferias_de']) && 
        !empty($dadosFerias['ferias_ate'])) {
        
        $dataAtual = new DateTime();
        $feriasDe = new DateTime($dadosFerias['ferias_de']);
        $feriasAte = new DateTime($dadosFerias['ferias_ate']);
        
        // Verifica se a data atual está dentro do período de férias
        if ($dataAtual >= $feriasDe && $dataAtual <= $feriasAte) {
            $fornecedorEmFerias = true;
            $feriasInfo = [
                'fornecedor' => $dadosFerias['nome'],
                'ferias_de' => date('d/m/Y', strtotime($dadosFerias['ferias_de'])),
                'ferias_ate' => date('d/m/Y', strtotime($dadosFerias['ferias_ate']))
            ];
        }
    }
}

// Query básica para contar o número total de produtos (para calcular a paginação)
$countSql = "SELECT COUNT(DISTINCT p.id) as total 
             FROM drophub_global.produtos p
             LEFT JOIN drophub_global.fornecedores f ON p.fornecedor = f.id 
             WHERE 1=1
             AND (p.is_filho IS NULL OR p.is_filho = 0)";


// Filtro por fornecedor
if (!empty($fornecedor)) {
    $countSql .= " AND f.nome = :fornecedor";
}

// Filtro por nome do produto
if (!empty($nomeProduto)) {
    $countSql .= " AND p.titulo LIKE :nomeProduto";
}

// Filtro por categoria interna
if (!empty($categoriaInterna)) {
    if ($categoriaInterna === 'Sem Estoque') {
        $countSql .= " AND p.estoque <= 0";
    } else {
        $countSql .= " AND p.categoria_interna = :categoriaInterna";
    }
}

$countStmt = $pdo->prepare($countSql);

// Bind de parâmetros
if (!empty($fornecedor)) {
    $countStmt->bindValue(':fornecedor', $fornecedor);
}

if (!empty($nomeProduto)) {
    $countStmt->bindValue(':nomeProduto', '%' . $nomeProduto . '%');
}

if (!empty($categoriaInterna) && $categoriaInterna !== 'Sem Estoque') {
    $countStmt->bindValue(':categoriaInterna', $categoriaInterna);
}

$countStmt->execute();
$totalItems = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];

// Query para buscar produtos com paginação
$sql = "SELECT p.id, p.sku, p.titulo, p.descricao, p.categoria_ml, p.codigo_universal, p.categoria_shopee, p.peso, p.altura, p.comprimento, p.ncm, p.largura, p.id_cat_ml, p.preco_custo, p.preco_minimo, p.estoque, p.add_data, 
       f.nome AS fornecedor_nome, p.sensivel
FROM drophub_global.produtos p 
LEFT JOIN drophub_global.fornecedores f ON p.fornecedor = f.id 
WHERE 1=1
AND (p.is_filho IS NULL OR p.is_filho = 0)";


// Filtro por fornecedor
if (!empty($fornecedor)) {
    $sql .= " AND f.nome = :fornecedor";
}

// Filtro por nome do produto
if (!empty($nomeProduto)) {
    $sql .= " AND (p.titulo LIKE :nomeProdutoTitulo OR p.sku LIKE :nomeProdutoSku OR p.sku_for LIKE :nomeProdutoSkuFor)";
}

// Filtro por categoria interna
if (!empty($categoriaInterna)) {
    if ($categoriaInterna === 'Sem Estoque') {
        $sql .= " AND p.estoque <= 0";
    } else {
        $sql .= " AND p.categoria_interna = :categoriaInterna";
    }
}

// Agrupar por produto para evitar duplicidades
$sql .= " GROUP BY p.id, p.sku, p.titulo, p.sensivel, p.descricao, p.categoria_ml, p.codigo_universal, p.categoria_shopee, p.peso, p.altura, p.comprimento, p.ncm, p.largura, p.id_cat_ml, p.preco_custo, p.preco_minimo, p.estoque, p.add_data, f.nome";

// Adicionar ordenação: primeiro pela quantidade de estoque e depois pela data de adição
$sql .= " ORDER BY 
        CASE WHEN p.estoque <= 1 THEN 1 ELSE 0 END, 
        p.add_data DESC";

// Limitar a busca pelos itens da página atual
$sql .= " LIMIT :limit OFFSET :offset";

$stmt = $pdo->prepare($sql);

// Bind de parâmetros
if (!empty($fornecedor)) {
    $stmt->bindValue(':fornecedor', $fornecedor);
}
if (!empty($nomeProduto)) {
    $stmt->bindValue(':nomeProdutoTitulo', '%' . $nomeProduto . '%');
    $stmt->bindValue(':nomeProdutoSku', '%' . $nomeProduto . '%');
    $stmt->bindValue(':nomeProdutoSkuFor', '%' . $nomeProduto . '%');
}
if (!empty($categoriaInterna) && $categoriaInterna !== 'Sem Estoque') {
    $stmt->bindValue(':categoriaInterna', $categoriaInterna);
}

// Bind para a paginação
$stmt->bindValue(':limit', $itemsPerPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);

$stmt->execute();
$produtos = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (!empty($produtos)) {
    $id_prod_base = $produtos[0]['id'];
    if (preg_match('/^([^_]+)/', $id_prod_base, $matches)) {
        $prefixo_forn = $matches[1] . '_';
    }
} else {
    $id_prod_base = null;
    $prefixo_forn = null;
}

?>

<?php
if ($produtos) {
    
    // Buscar categorias internas filtradas por fornecedor selecionado
    $categoriasSql = "SELECT DISTINCT p.categoria_interna 
                      FROM drophub_global.produtos p
                      LEFT JOIN drophub_global.fornecedores f ON p.fornecedor = f.id 
                      WHERE p.categoria_interna IS NOT NULL";
    
    // Se um fornecedor específico estiver selecionado, filtrar categorias apenas desse fornecedor
    if (!empty($fornecedor)) {
        $categoriasSql .= " AND f.nome = :fornecedor";
    }
    
    $categoriasSql .= " ORDER BY p.categoria_interna";
    
    $categoriasStmt = $pdo->prepare($categoriasSql);
    
    // Bind do parâmetro fornecedor se necessário
    if (!empty($fornecedor)) {
        $categoriasStmt->bindValue(':fornecedor', $fornecedor);
    }
    
    $categoriasStmt->execute();
    $categorias = $categoriasStmt->fetchAll(PDO::FETCH_ASSOC);

    
    // Título e Select2 alinhados
    echo '<div class="d-flex justify-content-between align-items-center mb-3">';
    echo '<h5 class="card-title mb-0">Catálogo de produtos</h5>';
    echo '<div class="d-flex align-items-center">';
    echo '<select id="categoria_interna" class="form-select select2" style="width: 300px;">';
    echo '<option value="">Selecione uma categoria interna...</option>';
    
    $selectedSemEstoque = ($categoriaInterna === 'Sem Estoque') ? 'selected' : '';
    echo '<option value="Sem Estoque" ' . $selectedSemEstoque . '>Sem Estoque</option>';

    foreach ($categorias as $categoria) {
        $selected = ($categoria['categoria_interna'] === $categoriaInterna) ? 'selected' : '';
        echo '<option value="' . htmlspecialchars($categoria['categoria_interna'], ENT_QUOTES, 'UTF-8') . '" ' . $selected . '>' . htmlspecialchars($categoria['categoria_interna'], ENT_QUOTES, 'UTF-8') . '</option>';
    }
    echo '</select>';
    echo '</div>';
    echo '</div>';

    echo '
    <script>
        $(document).ready(function() {
            $("#categoria_interna").select2({
                theme: "bootstrap-5",
                placeholder: "Selecione uma categoria interna...",
                allowClear: true
            });
        });
    </script>';

    // Obter nome do whitelabel para fallback
    $whitelabelNome = strtolower(Config::get('site_configs.name'));

    echo '<div class="row g-3">';
    foreach ($produtos as $produto) {
        $id_prod = $produto['id'];

        $queryAtribbutesShp = "SELECT * FROM drophub_global.produtos_atributos_shopee WHERE id_produto = :id_prod";
        $stmtAtribbutesShp = $pdo->prepare($queryAtribbutesShp);
        $stmtAtribbutesShp->bindParam(':id_prod', $id_prod, PDO::PARAM_INT);
        $stmtAtribbutesShp->execute();

        $blank = 0;

        while ($rowAtribbutesShp = $stmtAtribbutesShp->fetch(PDO::FETCH_ASSOC)) {
            if (
                empty($rowAtribbutesShp['attribute_value']) &&
                empty($rowAtribbutesShp['original_value_name']) &&
                (!isset($rowAtribbutesShp['value_id']) || (int)$rowAtribbutesShp['value_id'] === 0)
            ) {
                $blank = 1;
                break;
            }
        }

        // Verifica se o produto tem variações (filhos) com estoque
        $temVariacoesComEstoque = false;
        if ($produto['estoque'] <= 1) {
            $queryVariacoes = "SELECT COUNT(*) as total FROM drophub_global.produtos WHERE produto_pai = :produto_id AND is_filho = 1 AND estoque > 0";
            $stmtVariacoes = $pdo->prepare($queryVariacoes);
            $stmtVariacoes->bindParam(':produto_id', $produto['id'], PDO::PARAM_STR);
            $stmtVariacoes->execute();
            $resultVariacoes = $stmtVariacoes->fetch(PDO::FETCH_ASSOC);
            $temVariacoesComEstoque = ($resultVariacoes['total'] > 0);
        }

        $descricaoSegura = htmlspecialchars($produto['descricao'], ENT_QUOTES, 'UTF-8');
        
        // Usar o helper para buscar a thumbnail (com fallback automático)
        $imagemCapa = $imagemHelper->getThumbnailUrl($produto['id'], $whitelabelNome);

        if ($fornecedor == 'Selecione um fornecedor...') {
            echo 'Selecione um fornecedor...';
        } else {
            echo '
                <div class="col-12 col-sm-6 col-md-4 col-lg-3 col-xl-3">
                    <div class="card h-100 border ' . ($produto['sensivel'] == 1 ? 'border-danger' : '') . ' shadow-sm position-relative rounded">

                        <!-- Imagem centralizada -->
                        <div class="text-center p-3">
                            <img src="' . $imagemCapa . '" alt="' . htmlspecialchars($produto['titulo'], ENT_QUOTES, 'UTF-8') . '" 
                                 class="img-fluid rounded object-fit-cover" 
                                 width="120" height="120"
                                 loading="lazy">
                        </div>
                        
                        <!-- Conteúdo do card -->
                        <div class="card-body text-center p-3 d-flex flex-column">
                            <!-- Título do produto -->
                            <h6 class="card-title fw-semibold mb-2 small lh-sm">
                                ' . htmlspecialchars(limitarTexto($produto['titulo'], 50), ENT_QUOTES, 'UTF-8') . '
                            </h6>
                            
                            <!-- ID do produto -->
                            <p class="text-muted small mb-3">ID: ' . htmlspecialchars($produto['sku'], ENT_QUOTES, 'UTF-8') . '</p>
                            
                            <!-- Badges de status -->
                            <div class="mb-3">';
                            
                if ($produto['sensivel'] == 1) {
                    echo '<span class="badge bg-danger mb-1 me-1">Sensível</span>';
                }
                
                if ($produto['estoque'] <= 1) {
                    echo '<span class="badge bg-warning text-dark mb-1">Sem Estoque</span>';
                }
                            
            echo '      </div>
                            
                            <!-- Preço e Estoque -->
                            <div class="row g-2 mb-3">
                                <div class="col-6">
                                    <div class="text-center">
                                        <h6 class="text-primary mb-0 fw-bold">R$ ' . number_format($produto['preco_custo'], 2, ',', '.') . '</h6>
                                        <small class="text-muted">Preço</small>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="text-center">
                                        <h6 class="text-success mb-0 fw-bold">' . $produto['estoque'] . '</h6>
                                        <small class="text-muted">Estoque</small>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Botão de ação -->
                            <div class="mt-auto">';
                            
            if ($fornecedorEmFerias) {
                // Fornecedor de férias - botão desabilitado
                echo '<button class="btn btn-secondary btn-sm w-100" disabled title="Fornecedor em férias - não é possível cadastrar produtos">
                        <i class="ti ti-confetti me-2"></i>Férias
                      </button>';
            } elseif ($produto['estoque'] <= 1 && !$temVariacoesComEstoque) {
                echo '<button class="btn btn-outline-danger btn-sm w-100 disabled" disabled>
                        <i class="ti ti-alert-circle me-2"></i>Sem Estoque
                      </button>';
            } elseif ($produto['estoque'] <= 1 && $temVariacoesComEstoque) {
                // Produto pai sem estoque mas tem variações com estoque
                echo '<a href="/dash/cadastrar_prod?id=' . $produto['id'] . '" class="btn btn-outline-info btn-sm w-100">
                        <i class="ti ti-versions me-2"></i>Ver Variações
                      </a>';
            } elseif ($blank === 1) {
                echo '<button class="btn btn-outline-danger btn-sm w-100 disabled" disabled>
                        <i class="ti ti-alert-triangle me-2"></i>Erro nos Atributos
                      </button>';
            } else {
                echo '<button class="btn btn-outline-primary btn-sm w-100" onclick=\'cadastrarProduto(' . json_encode([
                    'descricao' => base64_encode($descricaoSegura),
                    'produto' => base64_encode(json_encode($produto))
                ]) . ', this)\'>
                        <i class="ti ti-plus me-2"></i>Cadastrar Produto
                      </button>';
            }
            
            echo '      </div>
                        </div>
                    </div>
                </div>';
        }
    }
    echo '</div>'; // FIM DO GRID
} else {
    echo '<p class="text-muted">Nenhum produto encontrado.</p>';
}

// Função JavaScript para filtrar por categoria interna
?>


<?php
$totalPages = ceil($totalItems / $itemsPerPage);

// Exibir paginação
if ($totalPages > 1) {
    echo '<nav aria-label="Page navigation">';
    echo '<ul class="pagination flex-wrap gap-2 justify-content-end">';

    // Link para a página anterior
    if ($page > 1) {
        echo '<li class="page-item"><a class="page-link" href="?page=' . ($page - 1) . '&categoria_interna=' . urlencode($categoriaInterna) . '&fornecedor=' . urlencode($fornecedor) . '&nomeProduto=' . urlencode($nomeProduto) . '">Anterior</a></li>';
    }

    // Links para cada página
    for ($i = 1; $i <= $totalPages; $i++) {
        $active = ($i == $page) ? 'active' : '';
        echo '<li class="page-item ' . $active . '"><a class="page-link" href="?page=' . $i . '&categoria_interna=' . urlencode($categoriaInterna) . '&fornecedor=' . urlencode($fornecedor) . '&nomeProduto=' . urlencode($nomeProduto) . '">' . $i . '</a></li>';
    }

    // Link para a próxima página
    if ($page < $totalPages) {
        echo '<li class="page-item"><a class="page-link" href="?page=' . ($page + 1) . '&categoria_interna=' . urlencode($categoriaInterna) . '&fornecedor=' . urlencode($fornecedor) . '&nomeProduto=' . urlencode($nomeProduto) . '">Próxima</a></li>';
    }

    echo '</ul>';
    echo '</nav>';
}
?>