<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Verifica se há uma sessão ativa
session_start();

if (!isset($_SESSION['usuario']) || !isset($_SESSION['usuario']['id_usuario'])) {
    echo json_encode(['success' => false, 'error' => 'Sessão inválida']);
    exit;
}

require '../../../class/configClass.php';
Config::load('../../../config/config.php');
include_once('../../../conexao/conexao.php');
require '../../../class/getTokens.php';

try {
    // Dados recebidos
    $produto_id_origem = trim($_POST['produto_id_origem'] ?? '');
    $loja_origem = trim($_POST['loja_origem'] ?? '');
    $plataforma_origem = trim($_POST['plataforma_origem'] ?? '');
    
    $produto_id = trim($_POST['produto_id'] ?? '');
    $usuario = trim($_POST['usuario'] ?? '');
    $id_usuario = trim($_POST['id_usuario'] ?? '');
    $loja = trim($_POST['loja'] ?? '');
    $titulo = trim($_POST['titulo'] ?? '');
    $descricao = trim($_POST['descricao'] ?? '');
    $sku = trim($_POST['sku'] ?? '');
    $sku_produto_selecionado = trim($_POST['sku_produto_selecionado'] ?? $sku); // Fallback para o SKU normal se não fornecido
    $codigo_universal = trim($_POST['codigo_universal'] ?? '');
    $estoque = intval($_POST['estoque'] ?? 0);
    $preco_custo = floatval($_POST['preco_custo'] ?? 0);
    $preco_sugerido = floatval($_POST['preco_sugerido'] ?? 0);
    $tipo_anuncio = trim($_POST['tipo_anuncio'] ?? '');
    $plataforma = trim($_POST['plataforma'] ?? '');
    $id_produto_ml = trim($_POST['id_produto_ml'] ?? '');
    $whitelabel = trim($_POST['whitelabel'] ?? '');
    
    // Se whitelabel não foi passado ou está vazio, usa o nome do banco
    if (empty($whitelabel)) {
        $whitelabel = Config::get('database.dbname');
    }

    // Validações obrigatórias
    if (empty($produto_id) || empty($titulo) || empty($sku) || empty($plataforma)) {
        echo json_encode(['success' => false, 'error' => 'Campos obrigatórios não preenchidos']);
        exit;
    }

    // Verifica se o usuário da sessão confere
    if ((int)$id_usuario !== (int)$_SESSION['usuario']['id_usuario']) {
        echo json_encode(['success' => false, 'error' => 'Usuário inválido']);
        exit;
    }

    // Verifica se o produto já não está cadastrado nesta plataforma/loja
    $sqlCheck = "SELECT id FROM drophub_global.produtos_cadastrados WHERE sku = ? AND plataforma = ? AND loja = ? AND id_usuario = ?";
    $stmtCheck = $pdo->prepare($sqlCheck);
    $stmtCheck->execute([$sku, $plataforma, $loja, $id_usuario]);
    
    if ($stmtCheck->fetch()) {
        echo json_encode(['success' => false, 'error' => 'Produto já cadastrado nesta plataforma/loja']);
        exit;
    }

    // Prepara a query de inserção
    $sql = "INSERT INTO drophub_global.produtos_cadastrados (
        produto_id, usuario, loja, id_usuario, titulo, descricao, sku, 
        codigo_universal, estoque, preco_custo, preco_sugerido, tipo_anuncio, 
        plataforma, id_produto_ml, whitelabel
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $params = [
        $produto_id,
        $usuario,
        $loja,
        $id_usuario,
        $titulo,
        $descricao,
        $sku,
        $codigo_universal,
        $estoque,
        $preco_custo,
        $preco_sugerido,
        $tipo_anuncio,
        $plataforma,
        $id_produto_ml,
        $whitelabel
    ];

    $stmt = $pdo->prepare($sql);
    
    // Debug: mostra os dados que serão inseridos
    error_log("DEBUG: Inserindo produto com dados: " . json_encode([
        'produto_id' => $produto_id,
        'usuario' => $usuario,
        'loja' => $loja,
        'id_usuario' => $id_usuario,
        'titulo' => $titulo,
        'sku' => $sku,
        'plataforma' => $plataforma,
        'whitelabel' => $whitelabel
    ]));
    
    if ($stmt->execute($params)) {
        $produto_cadastrado_id = $pdo->lastInsertId();
        
        // Busca e salva as imagens do produto original
        salvarImagensProdutoImportado($produto_cadastrado_id, $produto_id_origem, $loja_origem, $plataforma_origem, $pdo);
        
        $resultadoAtualizacao = atualizarSkuMercadoLivre($produto_id_origem, $loja_origem, $sku_produto_selecionado, $pdo);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Produto importado com sucesso',
            'id' => $produto_cadastrado_id
        ]);
    } else {
        $errorInfo = $stmt->errorInfo();
        error_log("Erro ao executar query de inserção: " . json_encode($errorInfo));
        echo json_encode(['success' => false, 'error' => 'Erro ao salvar produto: ' . $errorInfo[2]]);
    }

} catch (Exception $e) {
    error_log("Erro ao importar produto: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Erro interno do servidor: ' . $e->getMessage()]);
}

/**
 * Função para salvar as imagens do produto importado
 */
function salvarImagensProdutoImportado($produto_cadastrado_id, $produto_id_origem, $loja_origem, $plataforma_origem, $pdo) {
    try {
        // Converte nome da plataforma para path
        $plataformaPath = $plataforma_origem;
        if ($plataforma_origem === 'mercadolivre') {
            $plataformaPath = 'mercado_livre';
        }

        // Determina endpoint baseado na plataforma
        $endpoint = $plataformaPath === 'mercado_livre' ? 'get_prod_info_ml.php' : 'get_prod_info.php';
        
        // Inclui o arquivo diretamente para buscar as imagens
        $filePath = __DIR__ . '/plataformas/' . $plataformaPath . '/' . $endpoint;
        
        if (file_exists($filePath)) {
            // Simula parâmetros GET
            $_GET['id_ml'] = $produto_id_origem;
            $_GET['id_loja'] = $loja_origem;
            
            // Captura a saída
            ob_start();
            include $filePath;
            $response = ob_get_clean();
            
            $result = json_decode($response, true);
            
            if ($result && $result['success'] && isset($result['imagens']) && is_array($result['imagens'])) {
                // Salva cada imagem na tabela produtos_imagens
                foreach ($result['imagens'] as $index => $imagem) {
                    $imagemUrl = is_array($imagem) ? $imagem['url'] : $imagem;
                    
                    if (!empty($imagemUrl)) {
                        // Usa o ID do produto cadastrado (auto-increment) para as imagens
                        $sqlImg = "INSERT INTO produtos_imagens (produto_id, imagem_url, ordem, data_criacao) VALUES (?, ?, ?, NOW())";
                        $stmtImg = $pdo->prepare($sqlImg);
                        $resultado = $stmtImg->execute([$produto_cadastrado_id, $imagemUrl, $index + 1]);
                        
                        if (!$resultado) {
                            error_log("Erro ao salvar imagem: " . json_encode($stmtImg->errorInfo()));
                        }
                    }
                }
            }
        }
    } catch (Exception $e) {
        error_log("Erro ao salvar imagens do produto importado: " . $e->getMessage());
        // Não falha a importação por causa das imagens
    }
}

/**
 * Função para atualizar SKU e seller_custom_field no Mercado Livre
 */
function atualizarSkuMercadoLivre($produto_id_origem, $loja_origem, $sku, $pdo) {
    try {
        // Busca dados da loja para pegar o access_token
        $sql_loja = "SELECT seller_id, access_token, client_id, client_secret, code FROM usuarios_api_contas WHERE seller_id = :seller_id";
        $stmt_loja = $pdo->prepare($sql_loja);
        $stmt_loja->execute([':seller_id' => $loja_origem]);
        $loja = $stmt_loja->fetch(PDO::FETCH_ASSOC);
        
        if (!$loja) {
            return ['success' => false, 'error' => 'Loja não encontrada'];
        }
        
        // Refresh do token antes de fazer a atualização
        $refreshResult = refreshTokenML($loja, $pdo, $loja_origem);
        if (!$refreshResult['success']) {
            return ['success' => false, 'error' => 'Erro ao atualizar token: ' . $refreshResult['error']];
        }
        
        // Busca novamente os dados da loja com token atualizado
        $stmt_loja->execute([':seller_id' => $loja_origem]);
        $loja = $stmt_loja->fetch(PDO::FETCH_ASSOC);
        
        // Prepara dados para atualização no ML
        $updateData = [
            'seller_custom_field' => $sku . '#drophub',
        ];
                
        // Atualiza no Mercado Livre
        $url = "https://api.mercadolibre.com/items/{$produto_id_origem}";
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => "PUT",
            CURLOPT_POSTFIELDS => json_encode($updateData),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . $loja['access_token'],
                "Content-Type: application/json"
            ],
        ]);
        
        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        if ($httpCode == 200 && isset($responseData['id'])) {
            return ['success' => true, 'message' => 'SKU atualizado no Mercado Livre'];
        } else {
            $error = isset($responseData['message']) ? $responseData['message'] : 'Erro ao atualizar SKU no Mercado Livre';
            return ['success' => false, 'error' => $error, 'response' => $responseData];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => 'Erro interno ao atualizar SKU: ' . $e->getMessage()];
    }
}

/**
 * Função para fazer refresh do token do Mercado Livre
 */
function refreshTokenML($loja, $pdo, $seller_id_original) {
    try {
        $clientId = $loja['client_id'];
        $clientSecret = $loja['client_secret'];
        $refreshToken = $loja['code'];
        
        if (empty($clientId) || empty($clientSecret) || empty($refreshToken)) {
            return ['success' => false, 'error' => 'Dados da conta incompletos'];
        }
        
        // Faz refresh do token
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => 'https://api.mercadolibre.com/oauth/token',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => http_build_query(array(
                'grant_type' => 'refresh_token',
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
                'refresh_token' => $refreshToken
            )),
            CURLOPT_HTTPHEADER => array(
                'accept: application/json',
                'content-type: application/x-www-form-urlencoded'
            ),
        ));
        
        $responseRefresh = curl_exec($curl);
        $httpCodeRefresh = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        if ($httpCodeRefresh !== 200) {
            return ['success' => false, 'error' => 'Falha na comunicação com Mercado Livre para refresh token'];
        }
        
        $dataRefresh = json_decode($responseRefresh, true);
        $accessToken = $dataRefresh['access_token'] ?? null;
        $newRefreshToken = $dataRefresh['refresh_token'] ?? null;
        
        if (!$accessToken) {
            return ['success' => false, 'error' => 'Token de acesso não retornado pela API'];
        }
        
        // Atualiza o token no banco
        $sqlUpdate = "UPDATE usuarios_api_contas SET access_token = :access_token, code = :code WHERE seller_id = :seller_id";
        $stmtUpdate = $pdo->prepare($sqlUpdate);
        $updateResult = $stmtUpdate->execute([
            ':access_token' => $accessToken,
            ':code' => $newRefreshToken,
            ':seller_id' => $seller_id_original // Usar o seller_id original passado como parâmetro
        ]);
        
        if ($updateResult) {
            return ['success' => true, 'message' => 'Token atualizado com sucesso'];
        } else {
            return ['success' => false, 'error' => 'Falha ao atualizar token no banco de dados'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => 'Erro interno no refresh token: ' . $e->getMessage()];
    }
}
?>
