#!/usr/bin/env php
<?php
/**
 * Worker para processar fila de zeramento de estoque
 * 
 * Este worker processa jobs de zeramento de estoque de forma assíncrona,
 * com proteção contra processamento duplicado e suporte a retry automático.
 * 
 * Uso: php worker_zerar_estoque.php [--debug]
 * 
 * @author Sistema Drophub
 * @version 1.0
 * @date 2025-12-16
 */

// Configurações
$WORKER_NAME = 'worker_zerar_estoque';
$SLEEP_EMPTY_QUEUE = 5; // Segundos de espera quando fila vazia
$SLEEP_BETWEEN_JOBS = 1; // Segundos entre jobs
$LOCK_TIMEOUT = 300; // 5 minutos - tempo máximo de lock antes de considerar travado
$MAX_EXECUTION_TIME = 240; // 4 minutos - tempo máximo por job
$DEBUG = in_array('--debug', $argv ?? []);

// Paths
$BASE_PATH = dirname(__DIR__);
require_once $BASE_PATH . '/assets/class/configClass.php';
Config::load($BASE_PATH . '/assets/config/config.php');
require_once $BASE_PATH . '/assets/conexao/conexao.php';

// Arquivo de log
$LOG_FILE = $BASE_PATH . '/logs/worker_zerar_estoque.log';
$PID_FILE = $BASE_PATH . '/workers/worker_zerar_estoque.pid';

// Garantir que diretório de logs existe
if (!is_dir(dirname($LOG_FILE))) {
    mkdir(dirname($LOG_FILE), 0755, true);
}

/**
 * Função de log com timestamp
 */
function logMessage($message, $level = 'INFO') {
    global $LOG_FILE, $DEBUG;
    
    $timestamp = date('Y-m-d H:i:s');
    $pid = getmypid();
    $logLine = "[$timestamp] [$level] [PID:$pid] $message\n";
    
    file_put_contents($LOG_FILE, $logLine, FILE_APPEND);
    
    if ($DEBUG || $level === 'ERROR') {
        echo $logLine;
    }
}

/**
 * Registrar PID do worker
 */
function registrarPID() {
    global $PID_FILE;
    $pid = getmypid();
    file_put_contents($PID_FILE, $pid);
    logMessage("Worker iniciado com PID: $pid", 'INFO');
}

/**
 * Remover arquivo PID ao encerrar
 */
function removerPID() {
    global $PID_FILE;
    if (file_exists($PID_FILE)) {
        unlink($PID_FILE);
    }
    logMessage("Worker encerrado", 'INFO');
}

/**
 * Handler para sinais de término
 */
function signalHandler($signal) {
    logMessage("Recebido sinal $signal - encerrando gracefully...", 'WARN');
    removerPID();
    exit(0);
}

/**
 * Buscar próximo job disponível com lock atômico
 */
function buscarProximoJob($pdo) {
    global $LOCK_TIMEOUT;
    
    try {
        // Iniciar transação para garantir atomicidade
        $pdo->beginTransaction();
        
        // Liberar locks antigos (jobs travados por mais de LOCK_TIMEOUT)
        $stmtUnlock = $pdo->prepare("
            UPDATE drophub_global.fila_zerar_estoque 
            SET status = 'pending', 
                worker_pid = NULL, 
                locked_at = NULL,
                tentativas = tentativas + 1
            WHERE status = 'processing' 
            AND locked_at < DATE_SUB(NOW(), INTERVAL :timeout SECOND)
        ");
        $stmtUnlock->execute([':timeout' => $LOCK_TIMEOUT]);
        
        if ($stmtUnlock->rowCount() > 0) {
            logMessage("Liberados {$stmtUnlock->rowCount()} jobs travados", 'WARN');
        }
        
        // Buscar próximo job pendente e fazer lock atômico
        $stmt = $pdo->prepare("
            SELECT * FROM drophub_global.fila_zerar_estoque 
            WHERE status = 'pending' 
            AND tentativas < max_tentativas
            ORDER BY prioridade DESC, created_at ASC 
            LIMIT 1 
            FOR UPDATE
        ");
        $stmt->execute();
        $job = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$job) {
            $pdo->rollBack();
            return null;
        }
        
        // Marcar como processing e fazer lock
        $stmtUpdate = $pdo->prepare("
            UPDATE drophub_global.fila_zerar_estoque 
            SET status = 'processing',
                started_at = NOW(),
                locked_at = NOW(),
                worker_pid = :pid,
                tentativas = tentativas + 1
            WHERE id = :id
        ");
        $stmtUpdate->execute([
            ':id' => $job['id'],
            ':pid' => getmypid()
        ]);
        
        $pdo->commit();
        
        // Recarregar job atualizado
        $stmt = $pdo->prepare("SELECT * FROM drophub_global.fila_zerar_estoque WHERE id = :id");
        $stmt->execute([':id' => $job['id']]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        logMessage("Erro ao buscar próximo job: " . $e->getMessage(), 'ERROR');
        return null;
    }
}

/**
 * Processar job de zeramento
 */
function processarJob($pdo, $job) {
    global $MAX_EXECUTION_TIME, $BASE_PATH;
    
    $jobId = $job['id'];
    $sku = $job['sku'];
    $auto = $job['auto'];
    $estoqueOverride = $job['estoque_override'];
    
    logMessage("Iniciando processamento do job #$jobId - SKU: $sku", 'INFO');
    
    try {
        // Executar script em processo separado via PHP CLI
        $scriptPath = $BASE_PATH . '/assets/validations/zera_estoque_v5.php';
        
        // Construir query string
        $params = [
            'sku' => $sku,
            'auto' => $auto,
            'debug' => 0
        ];
        
        if ($estoqueOverride !== null) {
            $params['estoque_override'] = $estoqueOverride;
        }
        
        // Construir variáveis de ambiente para simular GET
        $getVars = '';
        foreach ($params as $key => $value) {
            $getVars .= "\$_GET['" . addslashes($key) . "'] = '" . addslashes($value) . "'; ";
        }
        
        // Executar PHP em processo separado com timeout
        // Redireciona stderr para /dev/null para suprimir warnings
        $cmd = sprintf(
            'cd %s && timeout %d php -d display_errors=0 -d error_reporting=0 -r %s 2>/dev/null',
            escapeshellarg(dirname($scriptPath)),
            $MAX_EXECUTION_TIME,
            escapeshellarg($getVars . ' include "' . basename($scriptPath) . '";')
        );
        
        logMessage("Executando: $cmd", 'DEBUG');
        
        $response = shell_exec($cmd);
        
        if ($response === null || empty($response)) {
            throw new Exception("Script não retornou resposta");
        }
        
        // Limpar resposta (remover espaços/quebras de linha extras)
        $response = trim($response);
        
        $resultado = json_decode($response, true);
        
        if (!$resultado) {
            throw new Exception("Resposta inválida da API: $response");
        }
        
        // Verificar se foi bem-sucedido
        if (isset($resultado['success']) && $resultado['success']) {
            // Marcar como completado
            $stmt = $pdo->prepare("
                UPDATE drophub_global.fila_zerar_estoque 
                SET status = 'completed',
                    completed_at = NOW(),
                    resultado_json = :resultado,
                    error_message = NULL,
                    worker_pid = NULL,
                    locked_at = NULL
                WHERE id = :id
            ");
            $stmt->execute([
                ':id' => $jobId,
                ':resultado' => json_encode($resultado, JSON_UNESCAPED_UNICODE)
            ]);
            
            logMessage("Job #$jobId concluído com sucesso - SKU: $sku - Zerados: {$resultado['total_zerados']}", 'INFO');
            return true;
            
        } else {
            // Falha no processamento
            $errorMsg = $resultado['error'] ?? 'Erro desconhecido';
            throw new Exception($errorMsg);
        }
        
    } catch (Exception $e) {
        $errorMsg = $e->getMessage();
        logMessage("Erro ao processar job #$jobId: $errorMsg", 'ERROR');
        
        // Verificar se deve marcar como failed ou voltar para pending
        if ($job['tentativas'] >= $job['max_tentativas'] - 1) {
            // Última tentativa - marcar como failed
            $stmt = $pdo->prepare("
                UPDATE drophub_global.fila_zerar_estoque 
                SET status = 'failed',
                    completed_at = NOW(),
                    error_message = :error,
                    worker_pid = NULL,
                    locked_at = NULL
                WHERE id = :id
            ");
            $stmt->execute([
                ':id' => $jobId,
                ':error' => $errorMsg
            ]);
            logMessage("Job #$jobId marcado como FAILED após {$job['tentativas']} tentativas", 'ERROR');
        } else {
            // Ainda tem tentativas - voltar para pending
            $stmt = $pdo->prepare("
                UPDATE drophub_global.fila_zerar_estoque 
                SET status = 'pending',
                    error_message = :error,
                    worker_pid = NULL,
                    locked_at = NULL
                WHERE id = :id
            ");
            $stmt->execute([
                ':id' => $jobId,
                ':error' => $errorMsg
            ]);
            logMessage("Job #$jobId voltou para PENDING - tentativa {$job['tentativas']}/{$job['max_tentativas']}", 'WARN');
        }
        
        return false;
    }
}

/**
 * Loop principal do worker
 */
function mainLoop($pdo) {
    global $SLEEP_EMPTY_QUEUE, $SLEEP_BETWEEN_JOBS, $WORKER_NAME;
    
    logMessage("Worker iniciado - aguardando jobs...", 'INFO');
    
    while (true) {
        try {
            // Buscar próximo job
            $job = buscarProximoJob($pdo);
            
            if (!$job) {
                // Fila vazia - aguardar
                if ($SLEEP_EMPTY_QUEUE > 0) {
                    sleep($SLEEP_EMPTY_QUEUE);
                }
                continue;
            }
            
            // Processar job
            processarJob($pdo, $job);
            
            // Aguardar entre jobs
            if ($SLEEP_BETWEEN_JOBS > 0) {
                sleep($SLEEP_BETWEEN_JOBS);
            }
            
        } catch (Exception $e) {
            logMessage("Erro no loop principal: " . $e->getMessage(), 'ERROR');
            sleep($SLEEP_EMPTY_QUEUE);
        }
    }
}

// ==================== INÍCIO DO SCRIPT ====================

// Registrar handlers de sinal
pcntl_signal(SIGTERM, 'signalHandler');
pcntl_signal(SIGINT, 'signalHandler');

// Verificar se já está rodando
if (file_exists($PID_FILE)) {
    $oldPid = (int)trim(file_get_contents($PID_FILE));
    if ($oldPid && posix_kill($oldPid, 0)) {
        logMessage("Worker já está rodando com PID: $oldPid", 'ERROR');
        die("Erro: Worker já está em execução (PID: $oldPid)\n");
    } else {
        logMessage("Removendo PID file antigo de processo morto: $oldPid", 'WARN');
        unlink($PID_FILE);
    }
}

// Registrar PID
registrarPID();

// Registrar shutdown function
register_shutdown_function('removerPID');

// Iniciar loop
try {
    mainLoop($pdo);
} catch (Exception $e) {
    logMessage("Erro fatal: " . $e->getMessage(), 'ERROR');
    removerPID();
    exit(1);
}
