#!/bin/bash
################################################################################
# Script de instalação do serviço systemd para Worker de Zeramento de Estoque
# 
# Este script configura o worker para rodar como serviço do sistema,
# iniciando automaticamente no boot e reiniciando em caso de falhas.
#
# Uso: sudo ./instalar_worker_zerar_estoque_service.sh
#
# @author Sistema Drophub
# @version 1.0
# @date 2025-12-16
################################################################################

# Verificar se está rodando como root
if [ "$EUID" -ne 0 ]; then 
    echo "ERRO: Este script deve ser executado como root (use sudo)"
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SERVICE_FILE="$SCRIPT_DIR/worker-zerar-estoque.service"
SERVICE_NAME="worker-zerar-estoque"
SYSTEMD_DIR="/etc/systemd/system"

echo "==================================================================="
echo "Instalador do Worker de Zeramento de Estoque - Systemd Service"
echo "==================================================================="
echo ""

# Verificar se arquivo de serviço existe
if [ ! -f "$SERVICE_FILE" ]; then
    echo "ERRO: Arquivo de serviço não encontrado: $SERVICE_FILE"
    exit 1
fi

echo "1. Parando serviço se estiver rodando..."
systemctl stop "$SERVICE_NAME" 2>/dev/null || true

echo "2. Copiando arquivo de serviço para $SYSTEMD_DIR..."
cp "$SERVICE_FILE" "$SYSTEMD_DIR/$SERVICE_NAME.service"

echo "3. Ajustando permissões..."
chmod 644 "$SYSTEMD_DIR/$SERVICE_NAME.service"

echo "4. Recarregando daemon do systemd..."
systemctl daemon-reload

echo "5. Habilitando serviço para iniciar no boot..."
systemctl enable "$SERVICE_NAME"

echo "6. Iniciando serviço..."
systemctl start "$SERVICE_NAME"

echo ""
echo "Aguardando 3 segundos..."
sleep 3

echo ""
echo "Status do serviço:"
echo "-------------------------------------------------------------------"
systemctl status "$SERVICE_NAME" --no-pager

echo ""
echo "==================================================================="
echo "Instalação concluída!"
echo "==================================================================="
echo ""
echo "Comandos úteis:"
echo "  sudo systemctl start $SERVICE_NAME     - Iniciar serviço"
echo "  sudo systemctl stop $SERVICE_NAME      - Parar serviço"
echo "  sudo systemctl restart $SERVICE_NAME   - Reiniciar serviço"
echo "  sudo systemctl status $SERVICE_NAME    - Ver status"
echo "  sudo journalctl -u $SERVICE_NAME -f    - Ver logs em tempo real"
echo "  sudo systemctl disable $SERVICE_NAME   - Desabilitar auto-start"
echo ""
echo "Logs:"
echo "  /var/www/whitelabels/stage/logs/worker_zerar_estoque.log"
echo "  /var/www/whitelabels/stage/logs/worker_zerar_estoque_error.log"
echo ""
