#!/bin/bash
################################################################################
# Script de diagnóstico para Worker de Zeramento de Estoque
# 
# Verifica configuração, status e possíveis problemas
#
# Uso: ./diagnosticar_worker_zerar_estoque.sh
#
# @author Sistema Drophub
# @version 1.0
# @date 2025-12-16
################################################################################

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BASE_DIR="$(dirname "$SCRIPT_DIR")"
WORKER_SCRIPT="$SCRIPT_DIR/worker_zerar_estoque.php"
PID_FILE="$SCRIPT_DIR/worker_zerar_estoque.pid"
LOG_FILE="$BASE_DIR/logs/worker_zerar_estoque.log"
API_SCRIPT="$SCRIPT_DIR/adicionar_job_zerar_estoque.php"
ZERA_ESTOQUE_SCRIPT="$BASE_DIR/assets/validations/zera_estoque_v5.php"

# Cores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_header() {
    echo ""
    echo -e "${BLUE}==================================================================="
    echo "$1"
    echo -e "===================================================================${NC}"
}

print_ok() {
    echo -e "  ${GREEN}✓${NC} $1"
}

print_warning() {
    echo -e "  ${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "  ${RED}✗${NC} $1"
}

# Header
clear
echo -e "${BLUE}"
echo "==================================================================="
echo "       DIAGNÓSTICO - Worker de Zeramento de Estoque"
echo "==================================================================="
echo -e "${NC}"

# 1. Verificar arquivos necessários
print_header "1. VERIFICAÇÃO DE ARQUIVOS"

if [ -f "$WORKER_SCRIPT" ]; then
    print_ok "Worker script encontrado: $WORKER_SCRIPT"
else
    print_error "Worker script NÃO encontrado: $WORKER_SCRIPT"
fi

if [ -f "$API_SCRIPT" ]; then
    print_ok "API script encontrado: $API_SCRIPT"
else
    print_error "API script NÃO encontrado: $API_SCRIPT"
fi

if [ -f "$ZERA_ESTOQUE_SCRIPT" ]; then
    print_ok "Script de zeramento encontrado: $ZERA_ESTOQUE_SCRIPT"
else
    print_error "Script de zeramento NÃO encontrado: $ZERA_ESTOQUE_SCRIPT"
fi

# 2. Verificar permissões
print_header "2. VERIFICAÇÃO DE PERMISSÕES"

if [ -x "$WORKER_SCRIPT" ]; then
    print_ok "Worker script é executável"
else
    print_warning "Worker script NÃO é executável (execute: chmod +x $WORKER_SCRIPT)"
fi

if [ -w "$(dirname "$LOG_FILE")" ]; then
    print_ok "Diretório de logs é gravável"
else
    print_error "Diretório de logs NÃO é gravável: $(dirname "$LOG_FILE")"
fi

# 3. Verificar PHP e extensões
print_header "3. VERIFICAÇÃO DO PHP"

if command -v php &> /dev/null; then
    PHP_VERSION=$(php -v | head -n 1)
    print_ok "PHP instalado: $PHP_VERSION"
else
    print_error "PHP NÃO encontrado"
fi

# Verificar extensões necessárias
php -m 2>/dev/null | grep -q pdo && print_ok "Extensão PDO instalada" || print_error "Extensão PDO NÃO instalada"
php -m 2>/dev/null | grep -q curl && print_ok "Extensão cURL instalada" || print_error "Extensão cURL NÃO instalada"
php -m 2>/dev/null | grep -q json && print_ok "Extensão JSON instalada" || print_error "Extensão JSON NÃO instalada"
php -m 2>/dev/null | grep -q pcntl && print_ok "Extensão PCNTL instalada" || print_error "Extensão PCNTL NÃO instalada"
php -m 2>/dev/null | grep -q posix && print_ok "Extensão POSIX instalada" || print_error "Extensão POSIX NÃO instalada"

# 4. Verificar banco de dados
print_header "4. VERIFICAÇÃO DO BANCO DE DADOS"

DB_CHECK=$(php -r "
    require_once '$BASE_DIR/assets/class/configClass.php';
    Config::load('$BASE_DIR/assets/config/config.php');
    try {
        require_once '$BASE_DIR/assets/conexao/conexao.php';
        echo 'OK';
    } catch (Exception \$e) {
        echo 'ERRO: ' . \$e->getMessage();
    }
" 2>&1)

if [ "$DB_CHECK" == "OK" ]; then
    print_ok "Conexão com banco de dados OK"
    
    # Verificar se tabela existe
    TABLE_EXISTS=$(php -r "
        require_once '$BASE_DIR/assets/class/configClass.php';
        Config::load('$BASE_DIR/assets/config/config.php');
        require_once '$BASE_DIR/assets/conexao/conexao.php';
        try {
            \$stmt = \$pdo->query('SELECT COUNT(*) FROM fila_zerar_estoque LIMIT 1');
            echo 'OK';
        } catch (Exception \$e) {
            echo 'ERRO';
        }
    " 2>&1)
    
    if [ "$TABLE_EXISTS" == "OK" ]; then
        print_ok "Tabela 'fila_zerar_estoque' existe"
        
        # Mostrar estatísticas
        php -r "
            require_once '$BASE_DIR/assets/class/configClass.php';
            Config::load('$BASE_DIR/assets/config/config.php');
            require_once '$BASE_DIR/assets/conexao/conexao.php';
            \$stmt = \$pdo->query('
                SELECT status, COUNT(*) as total 
                FROM fila_zerar_estoque 
                GROUP BY status
            ');
            echo '  Estatísticas da fila:\n';
            foreach (\$stmt->fetchAll(PDO::FETCH_ASSOC) as \$row) {
                printf('    - %s: %d jobs\n', strtoupper(\$row['status']), \$row['total']);
            }
        " 2>/dev/null
    else
        print_error "Tabela 'fila_zerar_estoque' NÃO existe"
        echo "    Execute: mysql < $SCRIPT_DIR/criar_tabela_fila_zerar_estoque.sql"
    fi
else
    print_error "Erro ao conectar no banco: $DB_CHECK"
fi

# 5. Status do worker
print_header "5. STATUS DO WORKER"

if [ -f "$PID_FILE" ]; then
    PID=$(cat "$PID_FILE")
    if ps -p "$PID" > /dev/null 2>&1; then
        print_ok "Worker está RODANDO (PID: $PID)"
        
        # Informações do processo
        echo ""
        echo "  Informações do processo:"
        ps -p "$PID" -o pid,ppid,user,%cpu,%mem,etime,cmd | tail -n +2 | sed 's/^/    /'
    else
        print_warning "PID file existe mas processo NÃO está rodando (PID morto: $PID)"
    fi
else
    print_warning "Worker NÃO está rodando (sem PID file)"
fi

# 6. Verificar serviço systemd
print_header "6. SERVIÇO SYSTEMD"

if systemctl list-unit-files | grep -q "worker-zerar-estoque.service"; then
    print_ok "Serviço systemd instalado"
    
    if systemctl is-enabled worker-zerar-estoque.service &>/dev/null; then
        print_ok "Serviço habilitado para auto-start"
    else
        print_warning "Serviço NÃO habilitado para auto-start"
    fi
    
    if systemctl is-active worker-zerar-estoque.service &>/dev/null; then
        print_ok "Serviço está ATIVO"
    else
        print_warning "Serviço NÃO está ativo"
    fi
else
    print_warning "Serviço systemd NÃO instalado"
    echo "    Execute: sudo $SCRIPT_DIR/instalar_worker_zerar_estoque_service.sh"
fi

# 7. Últimas linhas do log
print_header "7. ÚLTIMAS LINHAS DO LOG"

if [ -f "$LOG_FILE" ]; then
    echo ""
    tail -n 10 "$LOG_FILE" | sed 's/^/  /'
else
    print_warning "Arquivo de log não encontrado: $LOG_FILE"
fi

# 8. Recomendações
print_header "8. RECOMENDAÇÕES"

if [ ! -f "$PID_FILE" ] || ! ps -p "$(cat "$PID_FILE" 2>/dev/null)" > /dev/null 2>&1; then
    echo -e "  ${YELLOW}→${NC} Worker não está rodando. Para iniciar:"
    echo "    $SCRIPT_DIR/controlar_worker_zerar_estoque.sh start"
fi

if ! systemctl list-unit-files | grep -q "worker-zerar-estoque.service"; then
    echo -e "  ${YELLOW}→${NC} Para instalação permanente (auto-start no boot):"
    echo "    sudo $SCRIPT_DIR/instalar_worker_zerar_estoque_service.sh"
fi

echo -e "  ${YELLOW}→${NC} Para monitorar logs em tempo real:"
echo "    tail -f $LOG_FILE"

echo -e "  ${YELLOW}→${NC} Para adicionar job de teste:"
echo "    curl 'http://localhost/workers/adicionar_job_zerar_estoque.php?sku=TEST123&auto=1'"

echo ""
print_header "DIAGNÓSTICO CONCLUÍDO"
echo ""
