#!/bin/bash
################################################################################
# Script de controle do Worker de Zeramento de Estoque
# 
# Uso: ./controlar_worker_zerar_estoque.sh {start|stop|restart|status}
#
# Funções:
#   - start: Inicia o worker
#   - stop: Para o worker
#   - restart: Reinicia o worker
#   - status: Mostra status do worker
#
# @author Sistema Drophub
# @version 1.0
# @date 2025-12-16
################################################################################

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
WORKER_SCRIPT="$SCRIPT_DIR/worker_zerar_estoque.php"
PID_FILE="$SCRIPT_DIR/worker_zerar_estoque.pid"
LOG_FILE="$(dirname "$SCRIPT_DIR")/logs/worker_zerar_estoque.log"

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Função: Verificar se worker está rodando
is_running() {
    if [ -f "$PID_FILE" ]; then
        PID=$(cat "$PID_FILE")
        if ps -p "$PID" > /dev/null 2>&1; then
            return 0
        else
            # PID file existe mas processo não
            rm -f "$PID_FILE"
            return 1
        fi
    fi
    return 1
}

# Função: Iniciar worker
start_worker() {
    echo -n "Iniciando worker de zeramento de estoque... "
    
    if is_running; then
        echo -e "${YELLOW}Worker já está rodando (PID: $(cat "$PID_FILE"))${NC}"
        return 1
    fi
    
    # Garantir que diretório de logs existe
    mkdir -p "$(dirname "$LOG_FILE")"
    
    # Iniciar worker em background
    nohup php "$WORKER_SCRIPT" >> "$LOG_FILE" 2>&1 &
    
    # Aguardar criação do PID file
    sleep 2
    
    if is_running; then
        PID=$(cat "$PID_FILE")
        echo -e "${GREEN}OK${NC} (PID: $PID)"
        return 0
    else
        echo -e "${RED}FALHOU${NC}"
        echo "Verifique o log: $LOG_FILE"
        return 1
    fi
}

# Função: Parar worker
stop_worker() {
    echo -n "Parando worker de zeramento de estoque... "
    
    if ! is_running; then
        echo -e "${YELLOW}Worker não está rodando${NC}"
        return 1
    fi
    
    PID=$(cat "$PID_FILE")
    
    # Tentar parar gracefully (SIGTERM)
    kill -TERM "$PID" 2>/dev/null
    
    # Aguardar até 10 segundos
    for i in {1..10}; do
        if ! ps -p "$PID" > /dev/null 2>&1; then
            rm -f "$PID_FILE"
            echo -e "${GREEN}OK${NC}"
            return 0
        fi
        sleep 1
    done
    
    # Se não parou, forçar (SIGKILL)
    echo -n " (forçando) "
    kill -KILL "$PID" 2>/dev/null
    rm -f "$PID_FILE"
    echo -e "${GREEN}OK${NC}"
    return 0
}

# Função: Mostrar status
show_status() {
    echo "=== Status do Worker de Zeramento de Estoque ==="
    echo ""
    
    if is_running; then
        PID=$(cat "$PID_FILE")
        echo -e "Status: ${GREEN}RODANDO${NC}"
        echo "PID: $PID"
        
        # Mostrar informações do processo
        if command -v ps &> /dev/null; then
            echo ""
            echo "Informações do processo:"
            ps -p "$PID" -o pid,ppid,user,%cpu,%mem,etime,cmd 2>/dev/null || echo "  (não disponível)"
        fi
        
        # Mostrar estatísticas da fila
        echo ""
        echo "Estatísticas da fila:"
        php -r "
            require_once '$(dirname "$SCRIPT_DIR")/assets/class/configClass.php';
            Config::load('$(dirname "$SCRIPT_DIR")/assets/config/config.php');
            require_once '$(dirname "$SCRIPT_DIR")/assets/conexao/conexao.php';
            
            \$stmt = \$pdo->query(\"
                SELECT 
                    status,
                    COUNT(*) as total
                FROM fila_zerar_estoque
                GROUP BY status
            \");
            
            \$stats = \$stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty(\$stats)) {
                echo '  Fila vazia\n';
            } else {
                foreach (\$stats as \$stat) {
                    printf('  %s: %d\n', strtoupper(\$stat['status']), \$stat['total']);
                }
            }
        " 2>/dev/null || echo "  (erro ao consultar banco)"
        
    else
        echo -e "Status: ${RED}PARADO${NC}"
    fi
    
    # Mostrar últimas linhas do log
    if [ -f "$LOG_FILE" ]; then
        echo ""
        echo "Últimas 5 linhas do log:"
        tail -n 5 "$LOG_FILE"
    fi
    
    echo ""
}

# Função: Reiniciar worker
restart_worker() {
    echo "Reiniciando worker de zeramento de estoque..."
    stop_worker
    sleep 2
    start_worker
}

# Main
case "$1" in
    start)
        start_worker
        ;;
    stop)
        stop_worker
        ;;
    restart)
        restart_worker
        ;;
    status)
        show_status
        ;;
    *)
        echo "Uso: $0 {start|stop|restart|status}"
        echo ""
        echo "Comandos:"
        echo "  start   - Inicia o worker"
        echo "  stop    - Para o worker"
        echo "  restart - Reinicia o worker"
        echo "  status  - Mostra status do worker e da fila"
        exit 1
        ;;
esac

exit $?
