<?php
/**
 * API para adicionar jobs na fila de zeramento de estoque
 * 
 * Endpoint: /workers/adicionar_job_zerar_estoque.php
 * Método: POST ou GET
 * 
 * Parâmetros:
 *   - sku (obrigatório): SKU do produto a zerar
 *   - estoque_override (opcional): Estoque a restaurar (se não informado, usa valor do banco)
 *   - auto (opcional, padrão=1): Se deve restaurar automaticamente (1=sim, 0=não)
 *   - prioridade (opcional, padrão=0): Prioridade do job (maior = mais prioritário)
 * 
 * Resposta JSON:
 *   {
 *     "success": true|false,
 *     "job_id": 123,
 *     "message": "...",
 *     "queue_position": 5
 *   }
 * 
 * @author Sistema Drophub
 * @version 1.0
 * @date 2025-12-16
 */

header('Content-Type: application/json; charset=utf-8');

// Paths
$BASE_PATH = dirname(__DIR__);
require_once $BASE_PATH . '/assets/class/configClass.php';
Config::load($BASE_PATH . '/assets/config/config.php');
require_once $BASE_PATH . '/assets/conexao/conexao.php';

// Log
$LOG_FILE = $BASE_PATH . '/logs/fila_zerar_estoque_api.log';

function logAPI($message) {
    global $LOG_FILE;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($LOG_FILE, "[$timestamp] $message\n", FILE_APPEND);
}

/**
 * Validar e sanitizar parâmetros
 */
function validarParametros() {
    $params = [];
    
    // SKU (obrigatório)
    $sku = $_REQUEST['sku'] ?? '';
    $sku = trim($sku);
    
    if (empty($sku)) {
        return ['error' => 'Parâmetro "sku" é obrigatório'];
    }
    
    $params['sku'] = $sku;
    
    // Estoque override (opcional)
    if (isset($_REQUEST['estoque_override']) && $_REQUEST['estoque_override'] !== '') {
        $params['estoque_override'] = (int)$_REQUEST['estoque_override'];
    } else {
        $params['estoque_override'] = null;
    }
    
    // Auto (padrão = 1)
    $params['auto'] = isset($_REQUEST['auto']) ? (int)$_REQUEST['auto'] : 1;
    
    // Prioridade (padrão = 0)
    $params['prioridade'] = isset($_REQUEST['prioridade']) ? (int)$_REQUEST['prioridade'] : 0;
    
    return $params;
}

/**
 * Verificar se já existe job pendente ou em processamento para este SKU
 */
function verificarJobExistente($pdo, $sku) {
    $stmt = $pdo->prepare("
        SELECT id, status, tentativas, created_at 
        FROM drophub_global.fila_zerar_estoque 
        WHERE sku = :sku 
        AND status IN ('pending', 'processing')
        ORDER BY created_at DESC
        LIMIT 1
    ");
    $stmt->execute([':sku' => $sku]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Adicionar job na fila
 */
function adicionarJob($pdo, $params) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO drophub_global.fila_zerar_estoque 
            (sku, estoque_override, auto, prioridade, status, created_at)
            VALUES 
            (:sku, :estoque_override, :auto, :prioridade, 'pending', NOW())
        ");
        
        $stmt->execute([
            ':sku' => $params['sku'],
            ':estoque_override' => $params['estoque_override'],
            ':auto' => $params['auto'],
            ':prioridade' => $params['prioridade']
        ]);
        
        return $pdo->lastInsertId();
        
    } catch (PDOException $e) {
        logAPI("Erro ao inserir job: " . $e->getMessage());
        return false;
    }
}

/**
 * Contar posição na fila
 */
function contarPosicaoFila($pdo, $jobId, $prioridade) {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as posicao 
        FROM drophub_global.fila_zerar_estoque 
        WHERE status = 'pending'
        AND (
            prioridade > :prioridade1
            OR (prioridade = :prioridade2 AND id < :job_id)
        )
    ");
    $stmt->execute([
        ':prioridade1' => $prioridade,
        ':prioridade2' => $prioridade,
        ':job_id' => $jobId
    ]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return (int)$result['posicao'] + 1; // +1 porque é a posição (1-indexed)
}

// ==================== PROCESSAMENTO ====================

try {
    // Validar parâmetros
    $params = validarParametros();
    
    if (isset($params['error'])) {
        echo json_encode([
            'success' => false,
            'error' => $params['error']
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $sku = $params['sku'];
    
    logAPI("Requisição recebida - SKU: $sku | Auto: {$params['auto']} | Prioridade: {$params['prioridade']}");
    
    // Verificar se já existe job para este SKU
    $jobExistente = verificarJobExistente($pdo, $sku);
    
    if ($jobExistente) {
        logAPI("Job já existe para SKU $sku - ID: {$jobExistente['id']} | Status: {$jobExistente['status']}");
        
        echo json_encode([
            'success' => false,
            'error' => 'Já existe um job em processamento para este SKU',
            'job_existente' => [
                'id' => $jobExistente['id'],
                'status' => $jobExistente['status'],
                'tentativas' => $jobExistente['tentativas'],
                'created_at' => $jobExistente['created_at']
            ]
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Adicionar job na fila
    $jobId = adicionarJob($pdo, $params);
    
    if (!$jobId) {
        echo json_encode([
            'success' => false,
            'error' => 'Falha ao adicionar job na fila'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Contar posição na fila
    $posicao = contarPosicaoFila($pdo, $jobId, $params['prioridade']);
    
    logAPI("Job adicionado com sucesso - ID: $jobId | SKU: $sku | Posição: $posicao");
    
    echo json_encode([
        'success' => true,
        'job_id' => $jobId,
        'message' => 'Job adicionado à fila com sucesso',
        'sku' => $sku,
        'queue_position' => $posicao,
        'estimated_wait_seconds' => $posicao * 30 // Estimativa: 30s por job
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    logAPI("Erro: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'error' => 'Erro interno: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}
