<?php
session_start();
set_time_limit(180);

require '../class/configClass.php';
Config::load('../config/config.php');
include_once('../conexao/conexao.php');
include_once('../funcoes/general.php');

header('Content-Type: application/json; charset=utf-8');

// Função de log
function logDebug($msg) {
    $logFile = __DIR__ . '/zera_estoque_v5.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $msg\n", FILE_APPEND);
}

// ============================================================
// FUNÇÕES AUXILIARES PARA PARALELIZAÇÃO
// ============================================================

function prepararCurlZerarML($produtoML, $loja) {
    $itemId = $produtoML['id'];
    $status = $produtoML['status'] ?? 'active';
    
    // Se já está closed, não precisa zerar
    if ($status === 'closed') {
        return null;
    }
    
    $url = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
    $payload = json_encode(['available_quantity' => 0]);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_CUSTOMREQUEST => 'PUT',
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $loja['access_token']
        ]
    ]);
    
    return $ch;
}

function prepararCurlZerarShopee($produtoShopee, $loja) {
    $itemId = $produtoShopee['item_id'];
    $shopId = $loja['seller_id'];
    $accessToken = $loja['access_token'];
    $partnerId = $loja['client_id'];
    $partnerKey = $loja['client_secret'];
    
    $path = '/api/v2/product/update_stock';
    $timestamp = time();
    $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
    $sign = hash_hmac('sha256', $baseString, $partnerKey);
    
    $url = sprintf(
        'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s',
        $path, $partnerId, $shopId, $timestamp, $sign, $accessToken
    );
    
    $body = [
        'item_id' => (int)$itemId,
        'stock_list' => [
            [
                'model_id' => 0,
                'stock' => 0,
                'seller_stock' => [
                    [
                        'location' => 'default',
                        'stock' => 0
                    ]
                ]
            ]
        ]
    ];
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_POSTFIELDS => json_encode($body),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json']
    ]);
    
    return $ch;
}

function prepararCurlRestaurarML($produtoML, $loja, $produto) {
    $itemId = $produtoML['id'];
    $status = $produtoML['status'] ?? 'active';
    $estoqueReal = (int)$produto['estoque'];
    
    // Se está closed, mantém em 0
    if ($status === 'closed') {
        $quantidade = 0;
    } else {
        $quantidade = $estoqueReal;
        $status_novo = 'active';
    }
    
    $payload = ['available_quantity' => $quantidade];
    if (isset($status_novo)) {
        $payload['status'] = $status_novo;
    }
    
    $url = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_CUSTOMREQUEST => 'PUT',
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $loja['access_token']
        ]
    ]);
    
    return $ch;
}

function prepararCurlRestaurarShopee($produtoShopee, $loja, $produto) {
    $itemId = $produtoShopee['item_id'];
    $estoqueReal = (int)$produto['estoque'];
    $shopId = $loja['seller_id'];
    $accessToken = $loja['access_token'];
    $partnerId = $loja['client_id'];
    $partnerKey = $loja['client_secret'];
    
    $path = '/api/v2/product/update_stock';
    $timestamp = time();
    $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
    $sign = hash_hmac('sha256', $baseString, $partnerKey);
    
    $url = sprintf(
        'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s',
        $path, $partnerId, $shopId, $timestamp, $sign, $accessToken
    );
    
    $stockData = [
        'item_id' => $itemId,
        'stock_list' => [
            [
                'model_id' => 0,
                'stock' => $estoqueReal,
                'seller_stock' => [
                    [
                        'location' => 'default',
                        'stock' => $estoqueReal
                    ]
                ]
            ]
        ]
    ];
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_POSTFIELDS => json_encode($stockData),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json']
    ]);
    
    return $ch;
}

// Parâmetros
$sku = isset($_GET['sku']) ? trim($_GET['sku']) : '';
$debug = isset($_GET['debug']) && $_GET['debug'] == '1';
$etapa = isset($_GET['etapa']) ? (int)$_GET['etapa'] : 0;
$auto = isset($_GET['auto']) && $_GET['auto'] == '1';

// Captura estoque_override com validação robusta
$estoque_override = null;
if (isset($_GET['estoque_override']) && $_GET['estoque_override'] !== '') {
    $estoque_override = (int)$_GET['estoque_override'];
    logDebug("ESTOQUE OVERRIDE RECEBIDO: " . $_GET['estoque_override'] . " -> convertido para: " . $estoque_override);
}

if (empty($sku)) {
    echo json_encode(['success' => false, 'error' => 'SKU não informado']);
    exit;
}

$resultados = [
    'sku' => $sku,
    'fornecedor' => null,
    'lojas_integradas' => [],
    'total_zerados' => 0,
    'total_erros' => 0,
    'success' => false
];

if ($debug) {
    $resultados['debug'] = [];
}

try {
    // ========== ETAPA 1: BUSCAR PRODUTO E FORNECEDOR ==========
    $skuNormalizado = strtoupper(trim(explode('#', $sku)[0]));
    
    $stmt_produto = $pdo->prepare("
        SELECT p.id, p.sku, p.fornecedor, f.nome AS fornecedor_nome
        FROM drophub_global.produtos p
        LEFT JOIN drophub_global.fornecedores f ON p.fornecedor = f.id
        WHERE UPPER(TRIM(SUBSTRING_INDEX(p.sku, '#', 1))) = :sku
        LIMIT 1
    ");
    $stmt_produto->execute([':sku' => $skuNormalizado]);
    $produto = $stmt_produto->fetch(PDO::FETCH_ASSOC);
    
    if ($debug) {
        $resultados['debug']['sku_normalizado'] = $skuNormalizado;
        $resultados['debug']['produto_encontrado'] = $produto ? 'Sim' : 'Não';
        if ($produto) {
            $resultados['debug']['estoque_banco_original'] = $produto['estoque'] ?? 'NULL';
        }
    }
    
    // Usa estoque_override se fornecido, caso contrário usa o valor do banco
    if ($produto && $estoque_override !== null) {
        logDebug("ANTES: produto[estoque] = " . ($produto['estoque'] ?? 'NULL'));
        $produto['estoque'] = $estoque_override;
        logDebug("DEPOIS: produto[estoque] = " . $produto['estoque']);
        
        if ($debug) {
            $resultados['debug']['estoque_override_usado'] = $estoque_override;
            $resultados['debug']['estoque_final'] = $produto['estoque'];
        }
    }
    
    if (!$produto) {
        $resultados['error'] = 'Produto não encontrado na base de dados';
        echo json_encode($resultados, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    if (!$produto['fornecedor']) {
        $resultados['error'] = 'Produto não possui fornecedor vinculado';
        echo json_encode($resultados, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $resultados['fornecedor'] = [
        'id' => $produto['fornecedor'],
        'nome' => $produto['fornecedor_nome']
    ];
    
    if ($debug) {
        $resultados['debug']['produto_id'] = $produto['id'];
        $resultados['debug']['produto_sku'] = $produto['sku'];
    }
    
    // ========== ETAPA 2: BUSCAR LOJAS INTEGRADAS COM O FORNECEDOR ==========
    $stmt_lojas = $pdo->prepare("
        SELECT 
            seller_id,
            access_token,
            code AS refresh_token,
            client_id,
            client_secret,
            plataforma,
            whitelabel,
            user_id,
            fornecedor
        FROM drophub_global.usuarios_api_contas
        WHERE fornecedor = :fornecedor
        AND plataforma IN ('mercadolivre', 'mercado_livre', 'mercado-livre', 'shopee')
    ");
    $stmt_lojas->execute([':fornecedor' => $produto['fornecedor']]);
    $lojas = $stmt_lojas->fetchAll(PDO::FETCH_ASSOC);
    
    if ($debug) {
        $resultados['debug']['total_lojas_integradas'] = count($lojas);
    }
    
    if (empty($lojas)) {
        $resultados['error'] = 'Nenhuma loja integrada encontrada para este fornecedor';
        echo json_encode($resultados, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // ========== ETAPA 3: PROCESSAR LOJAS COM CACHE + PARALELIZAÇÃO ==========
    
    // CACHE: Buscar produtos de todas as lojas UMA VEZ
    $cache_produtos = [];
    
    if ($debug) {
        logDebug("=== INICIANDO CACHE DE PRODUTOS ===");
    }
    
    foreach ($lojas as $loja) {
        $plataforma = strtolower($loja['plataforma']);
        if (in_array($plataforma, ['mercado_livre', 'mercado-livre'])) {
            $plataforma = 'mercadolivre';
        }
        
        $cache_key = $plataforma . '|' . $loja['seller_id'];
        
        try {
            if ($plataforma === 'mercadolivre') {
                $cache_produtos[$cache_key] = buscarTodosProdutosML($skuNormalizado, $loja['access_token'], $loja['seller_id'], $debug);
                if ($debug) {
                    logDebug("Cache ML {$loja['seller_id']}: " . count($cache_produtos[$cache_key]) . " produtos");
                }
            } elseif ($plataforma === 'shopee') {
                $cache_produtos[$cache_key] = buscarProdutoShopee($skuNormalizado, $loja['access_token'], $loja['seller_id'], $loja['client_id'], $loja['client_secret'], $debug);
                if ($debug) {
                    logDebug("Cache Shopee {$loja['seller_id']}: " . ($cache_produtos[$cache_key] ? 'encontrado' : 'não encontrado'));
                }
            }
        } catch (Exception $e) {
            $cache_produtos[$cache_key] = null;
            if ($debug) {
                logDebug("Erro no cache {$cache_key}: " . $e->getMessage());
            }
        }
    }
    
    // PARALELIZAÇÃO: Processar lotes de 10 lojas simultaneamente
    $tamanho_lote = 10;
    $lotes = array_chunk($lojas, $tamanho_lote);
    
    if ($debug) {
        logDebug("=== PROCESSANDO " . count($lojas) . " LOJAS EM " . count($lotes) . " LOTES DE {$tamanho_lote} ===");
    }
    
    foreach ($lotes as $indice_lote => $lote) {
        if ($debug) {
            logDebug("Iniciando lote " . ($indice_lote + 1) . "/" . count($lotes));
        }
        
        $mh = curl_multi_init();
        $handles = [];
        $handle_map = [];
        
        foreach ($lote as $indice_loja => $loja) {
            $plataforma = strtolower($loja['plataforma']);
            if (in_array($plataforma, ['mercado_livre', 'mercado-livre'])) {
                $plataforma = 'mercadolivre';
            }
            
            $cache_key = $plataforma . '|' . $loja['seller_id'];
            $produtos_encontrados = $cache_produtos[$cache_key] ?? null;
            
            // Prepara requisição de zerar baseado nos produtos do cache
            if ($plataforma === 'mercadolivre' && !empty($produtos_encontrados)) {
                foreach ($produtos_encontrados as $produtoML) {
                    $ch = prepararCurlZerarML($produtoML, $loja);
                    if ($ch) {
                        curl_multi_add_handle($mh, $ch);
                        $handles[] = $ch;
                        $handle_map[(int)$ch] = [
                            'loja' => $loja,
                            'plataforma' => $plataforma,
                            'item_id' => $produtoML['id'],
                            'status' => $produtoML['status']
                        ];
                    }
                }
            } elseif ($plataforma === 'shopee' && $produtos_encontrados) {
                $ch = prepararCurlZerarShopee($produtos_encontrados, $loja);
                if ($ch) {
                    curl_multi_add_handle($mh, $ch);
                    $handles[] = $ch;
                    $handle_map[(int)$ch] = [
                        'loja' => $loja,
                        'plataforma' => $plataforma,
                        'item_id' => $produtos_encontrados['item_id']
                    ];
                }
            }
        }
        
        if ($debug) {
            logDebug("Lote " . ($indice_lote + 1) . ": " . count($handles) . " handles preparados");
        }
        
        // Executar todas requisições do lote em paralelo
        if (count($handles) > 0) {
            if ($debug) {
                logDebug("Executando curl_multi para " . count($handles) . " requisições");
            }
            
            $running = null;
            do {
                curl_multi_exec($mh, $running);
                curl_multi_select($mh);
            } while ($running > 0);
            
            if ($debug) {
                logDebug("curl_multi finalizado");
            }
        }
        
        // Processar respostas
        foreach ($handles as $ch) {
            $info = curl_getinfo($ch);
            $response = curl_multi_getcontent($ch);
            $handle_id = (int)$ch;
            $dados = $handle_map[$handle_id] ?? null;
            
            if ($dados) {
                $loja_resultado = [
                    'plataforma' => $dados['plataforma'],
                    'seller_id' => $dados['loja']['seller_id'],
                    'whitelabel' => $dados['loja']['whitelabel'],
                    'encontrado' => true,
                    'zerado' => ($info['http_code'] >= 200 && $info['http_code'] < 300),
                    'error' => null
                ];
                
                if ($loja_resultado['zerado']) {
                    $resultados['total_zerados']++;
                } else {
                    $loja_resultado['error'] = 'HTTP ' . $info['http_code'];
                    $resultados['total_erros']++;
                }
                
                $resultados['lojas_integradas'][] = $loja_resultado;
            }
            
            curl_multi_remove_handle($mh, $ch);
            curl_close($ch);
        }
        
        curl_multi_close($mh);
        
        if ($debug) {
            logDebug("Lote " . ($indice_lote + 1) . "/" . count($lotes) . " processado: " . count($handles) . " requisições");
        }
    }
    
    $resultados['success'] = $resultados['total_zerados'] > 0;
    
    // ========== ETAPA 4: RESTAURAR ESTOQUE COM PARALELIZAÇÃO (SE AUTO=1) ==========
    if ($auto && $resultados['success']) {
        if ($debug) {
            logDebug("=== RESTAURAR ESTOQUE ===");
            logDebug('Aguardando 2 segundos antes de restaurar...');
        }
        
        // Aguarda 2 segundos antes de restaurar
        sleep(2);
        
        $resultados['restauracao'] = [
            'iniciada' => true,
            'total_restaurados' => 0,
            'total_erros_restauracao' => 0,
            'lojas' => []
        ];
        
        // Processar restauração em lotes paralelos
        $lojas_para_restaurar = [];
        foreach ($resultados['lojas_integradas'] as $loja_zerada) {
            if ($loja_zerada['zerado']) {
                // Busca loja original
                foreach ($lojas as $l) {
                    if ($l['seller_id'] === $loja_zerada['seller_id']) {
                        $lojas_para_restaurar[] = [
                            'loja' => $l,
                            'plataforma' => $loja_zerada['plataforma']
                        ];
                        break;
                    }
                }
            }
        }
        
        // Processar em lotes de 10
        $lotes_restauracao = array_chunk($lojas_para_restaurar, 10);
        
        foreach ($lotes_restauracao as $indice_lote => $lote) {
            $mh = curl_multi_init();
            $handles = [];
            $handle_map = [];
            
            foreach ($lote as $item) {
                $loja = $item['loja'];
                $plataforma = $item['plataforma'];
                $cache_key = $plataforma . '|' . $loja['seller_id'];
                $produtos_cache = $cache_produtos[$cache_key] ?? null;
                
                if ($plataforma === 'mercadolivre' && !empty($produtos_cache)) {
                    foreach ($produtos_cache as $produtoML) {
                        $ch = prepararCurlRestaurarML($produtoML, $loja, $produto);
                        if ($ch) {
                            curl_multi_add_handle($mh, $ch);
                            $handles[] = $ch;
                            $handle_map[(int)$ch] = [
                                'loja' => $loja,
                                'plataforma' => $plataforma,
                                'seller_id' => $loja['seller_id']
                            ];
                        }
                    }
                } elseif ($plataforma === 'shopee' && $produtos_cache) {
                    $ch = prepararCurlRestaurarShopee($produtos_cache, $loja, $produto);
                    if ($ch) {
                        curl_multi_add_handle($mh, $ch);
                        $handles[] = $ch;
                        $handle_map[(int)$ch] = [
                            'loja' => $loja,
                            'plataforma' => $plataforma,
                            'seller_id' => $loja['seller_id']
                        ];
                    }
                }
            }
            
            // Executar em paralelo
            $running = null;
            do {
                curl_multi_exec($mh, $running);
                curl_multi_select($mh);
            } while ($running > 0);
            
            // Processar respostas
            $sellers_processados = [];
            foreach ($handles as $ch) {
                $info = curl_getinfo($ch);
                $handle_id = (int)$ch;
                $dados = $handle_map[$handle_id] ?? null;
                
                if ($dados && !in_array($dados['seller_id'], $sellers_processados)) {
                    $loja_restauracao = [
                        'plataforma' => $dados['plataforma'],
                        'seller_id' => $dados['seller_id'],
                        'restaurado' => ($info['http_code'] >= 200 && $info['http_code'] < 300),
                        'error' => null
                    ];
                    
                    if ($loja_restauracao['restaurado']) {
                        $resultados['restauracao']['total_restaurados']++;
                    } else {
                        $loja_restauracao['error'] = 'HTTP ' . $info['http_code'];
                        $resultados['restauracao']['total_erros_restauracao']++;
                    }
                    
                    $resultados['restauracao']['lojas'][] = $loja_restauracao;
                    $sellers_processados[] = $dados['seller_id'];
                }
                
                curl_multi_remove_handle($mh, $ch);
                curl_close($ch);
            }
            
            curl_multi_close($mh);
            
            if ($debug) {
                logDebug("Lote restauração " . ($indice_lote + 1) . "/" . count($lotes_restauracao) . " processado");
            }
        }
        
        $resultados['restauracao']['sucesso'] = $resultados['restauracao']['total_restaurados'] > 0;
    }
    
} catch (Exception $e) {
    $resultados['error'] = $e->getMessage();
    $resultados['success'] = false;
}

echo json_encode($resultados, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

// ==================== FUNÇÕES AUXILIARES ====================

// ============================================================
// FUNÇÃO DE PROCESSAMENTO MERCADO LIVRE (LEGADO)
// ============================================================

function processarMercadoLivre($loja, $skuNormalizado, $debug = false) {
    if ($debug) {
        logDebug("=== PROCESSAR MERCADO LIVRE ===");
        logDebug("Seller ID: {$loja['seller_id']} | SKU: $skuNormalizado");
    }
    
    $resultado = [
        'encontrado' => false,
        'zerado' => false,
        'produto_id' => null,
        'produtos_encontrados' => 0,
        'produtos_zerados' => 0,
        'error' => null
    ];
    
    if (empty($loja['seller_id']) || empty($loja['access_token'])) {
        $resultado['error'] = 'Credenciais incompletas';
        return $resultado;
    }
    
    // Buscar TODOS os produtos com esse SKU
    $produtos = buscarTodosProdutosML($skuNormalizado, $loja['access_token'], $loja['seller_id'], $debug);
    
    if ($debug) {
        logDebug("ML - Produtos encontrados: " . count($produtos));
        logDebug("ML - IDs: " . json_encode(array_column($produtos, 'id')));
    }
    
    if (empty($produtos)) {
        $resultado['error'] = 'Produto não encontrado nesta loja';
        return $resultado;
    }
    
    $resultado['encontrado'] = true;
    $resultado['produtos_encontrados'] = count($produtos);
    $erros = [];
    
    // Zerar estoque de cada produto encontrado
    foreach ($produtos as $produto) {
        if ($debug) {
            logDebug("ML - Zerando produto: {$produto['id']} | Status: {$produto['status']}");
        }
        
        $zero_result = zerarEstoqueML($produto, $loja['access_token']);
        
        if ($debug) {
            logDebug("ML - Resultado zerar {$produto['id']}: " . json_encode($zero_result));
        }
        
        if ($zero_result['success']) {
            $resultado['produtos_zerados']++;
        } else {
            $erros[] = $produto['id'] . ': ' . ($zero_result['error'] ?? 'Erro desconhecido');
        }
    }
    
    if ($resultado['produtos_zerados'] > 0) {
        $resultado['zerado'] = true;
        $resultado['produto_id'] = implode(', ', array_column($produtos, 'id'));
        
        if (!empty($erros)) {
            $resultado['error'] = 'Alguns produtos falharam: ' . implode('; ', $erros);
        }
    } else {
        $resultado['error'] = 'Falha ao zerar todos: ' . implode('; ', $erros);
    }
    
    return $resultado;
}

function processarShopee($loja, $skuNormalizado, $debug = false) {
    if ($debug) {
        logDebug("=== PROCESSAR SHOPEE ===");
        logDebug("Loja: " . json_encode($loja));
    }
    
    $resultado = [
        'encontrado' => false,
        'zerado' => false,
        'item_id' => null,
        'error' => null
    ];
    
    // Para Shopee, seller_id é o shop_id
    $shopId = $loja['seller_id'] ?? null;
    
    if (empty($shopId) || empty($loja['access_token']) || empty($loja['client_id']) || empty($loja['client_secret'])) {
        $resultado['error'] = 'Credenciais incompletas';
        return $resultado;
    }
    
    if ($debug) {
        $resultado['debug_shopee'] = [
            'shop_id' => $shopId,
            'partner_id' => $loja['client_id'],
            'sku_busca' => $skuNormalizado
        ];
    }
    
    // Buscar produto
    $produto = buscarProdutoShopee(
        $skuNormalizado, 
        $loja['access_token'], 
        $shopId, 
        $loja['client_id'], 
        $loja['client_secret'], 
        $debug
    );
    
    if ($debug && $produto) {
        $resultado['debug_shopee']['produto_encontrado'] = [
            'item_id' => $produto['item_id'] ?? null,
            'item_sku' => $produto['item_sku'] ?? null
        ];
    }
    
    if (!$produto) {
        $resultado['error'] = 'Produto não encontrado nesta loja';
        return $resultado;
    }
    
    $resultado['encontrado'] = true;
    $resultado['item_id'] = $produto['item_id'];
    
    // Zerar estoque (passa shopId explicitamente)
    $zero_result = zerarEstoqueShopee(
        $produto, 
        $loja['access_token'], 
        $loja['client_id'], 
        $loja['client_secret'],
        $shopId
    );
    
    if ($debug) {
        $resultado['debug_shopee']['zero_result'] = $zero_result;
    }
    
    if ($zero_result['success']) {
        $resultado['zerado'] = true;
    } else {
        $resultado['error'] = $zero_result['error'] ?? 'Falha ao zerar estoque';
    }
    
    return $resultado;
}

function buscarTodosProdutosML($sku, $accessToken, $sellerId, $debug = false) {
    $skuBusca = strtoupper(trim(explode('#', $sku)[0]));
    $produtosEncontrados = [];
    $offset = 0;
    $limit = 50;
    $totalVerificados = 0;
    
    if ($debug) {
        logDebug("ML buscarTodosProdutos - Iniciando busca por SKU: $skuBusca");
    }
    
    while ($offset < 1000) { // Aumentado para 1000
        $url = "https://api.mercadolibre.com/users/{$sellerId}/items/search?offset={$offset}&limit={$limit}";
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $accessToken
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            if ($debug) {
                logDebug("ML buscarTodosProdutos - HTTP {$httpCode} em offset $offset, parando");
            }
            break;
        }
        
        $data = json_decode($response, true);
        if (empty($data['results'])) {
            if ($debug) {
                logDebug("ML buscarTodosProdutos - Sem mais resultados em offset $offset");
            }
            break;
        }
        
        $ids = $data['results']; // Pega TODOS os IDs, não só 20
        $totalVerificados += count($ids);
        
        if ($debug) {
            logDebug("ML buscarTodosProdutos - Offset $offset: " . count($ids) . " IDs para verificar");
        }
        
        if (empty($ids)) {
            break;
        }
        
        // Processa em lotes de 20 (limite do ML para busca batch)
        foreach (array_chunk($ids, 20) as $chunk) {
            $idsStr = implode(',', $chunk);
            $urlBatch = "https://api.mercadolibre.com/items?ids={$idsStr}&attributes=id,seller_custom_field,variations,status,available_quantity";
            
            $ch = curl_init($urlBatch);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    'Authorization: Bearer ' . $accessToken
                ]
            ]);
            
            $batchResponse = curl_exec($ch);
            curl_close($ch);
            
            $items = json_decode($batchResponse, true);
            if (!is_array($items)) {
                continue;
            }
            
            foreach ($items as $item) {
                if (!isset($item['body'])) continue;
                $produto = $item['body'];
                
                $match = false;
                
                // Verifica SKU principal
                if (isset($produto['seller_custom_field'])) {
                    $itemSku = strtoupper(trim(explode('#', $produto['seller_custom_field'])[0]));
                    if ($itemSku === $skuBusca) {
                        $match = true;
                    }
                }
                
                // Verifica variações
                if (!$match && isset($produto['variations']) && is_array($produto['variations'])) {
                    foreach ($produto['variations'] as $variation) {
                        if (isset($variation['seller_custom_field'])) {
                            $varSku = strtoupper(trim(explode('#', $variation['seller_custom_field'])[0]));
                            if ($varSku === $skuBusca) {
                                $match = true;
                                break;
                            }
                        }
                    }
                }
                
                // Se encontrou match, busca detalhes completos e adiciona à lista
                if ($match) {
                    $urlFull = "https://api.mercadolibre.com/items/{$produto['id']}";
                    $chFull = curl_init($urlFull);
                    curl_setopt_array($chFull, [
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_HTTPHEADER => [
                            'Authorization: Bearer ' . $accessToken
                        ]
                    ]);
                    $fullResponse = curl_exec($chFull);
                    curl_close($chFull);
                    
                    $fullProduto = json_decode($fullResponse, true);
                    $produtosEncontrados[] = $fullProduto ?: $produto;
                }
            }
        }
        
        $offset += $limit;
    }
    
    if ($debug) {
        logDebug("ML buscarTodosProdutos - Total produtos verificados: $totalVerificados, Encontrados com SKU: " . count($produtosEncontrados));
    }
    
    return $produtosEncontrados;
}

function buscarProdutoML($sku, $accessToken, $sellerId, $debug = false) {
    $todos = buscarTodosProdutosML($sku, $accessToken, $sellerId, $debug);
    return !empty($todos) ? $todos[0] : null;
}

function buscarProdutoShopee($sku, $accessToken, $shopId, $partnerId, $partnerKey, $debug = false) {
    $skuBusca = strtoupper(trim(explode('#', $sku)[0]));
    
    if ($debug) {
        logDebug("=== INÍCIO BUSCA SHOPEE ===");
        logDebug("SKU Busca: $skuBusca | Shop ID: $shopId | Partner ID: $partnerId");
    }
    
    $debugInfo = [
        'total_items_found' => 0,
        'total_items_checked' => 0,
        'sample_skus' => []
    ];
    
    $allItemIds = [];
    
    // Busca até 10 páginas (1000 produtos) para garantir que encontra
    for ($page = 0; $page < 10; $page++) {
        $offset = $page * 100;
        $path = '/api/v2/product/get_item_list';
        $timestamp = time();
        $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
        $sign = hash_hmac('sha256', $baseString, $partnerKey);
        
        $url = sprintf(
            'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s&offset=%d&page_size=100&item_status=NORMAL',
            $path, $partnerId, $shopId, $timestamp, $sign, $accessToken, $offset
        );
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json']
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($debug && $page === 0) {
            $debugInfo['first_list_http_code'] = $httpCode;
            $debugInfo['first_list_response_preview'] = substr($response, 0, 500);
        }
        
        if ($httpCode !== 200) {
            $debugInfo['stopped_at_page'] = $page;
            $debugInfo['last_http_code'] = $httpCode;
            break;
        }
        
        $data = json_decode($response, true);
        
        if (!isset($data['response']['item']) || !is_array($data['response']['item']) || empty($data['response']['item'])) {
            $debugInfo['stopped_at_page'] = $page;
            $debugInfo['reason'] = 'no_more_items';
            break;
        }
        
        $debugInfo['total_items_found'] += count($data['response']['item']);
        
        foreach ($data['response']['item'] as $item) {
            if (isset($item['item_id'])) {
                $allItemIds[] = $item['item_id'];
            }
        }
        
        if (count($data['response']['item']) < 100) {
            $debugInfo['stopped_at_page'] = $page;
            $debugInfo['reason'] = 'less_than_100_items';
            break;
        }
    }
    
    if (empty($allItemIds)) {
        if ($debug) {
            logDebug('Shopee busca debug (sem itens): ' . json_encode($debugInfo));
        }
        return null;
    }
    
    $debugInfo['total_item_ids'] = count($allItemIds);
    
    // Processa em lotes de 50 itens
    foreach (array_chunk($allItemIds, 50) as $chunkIndex => $chunk) {
        $path = '/api/v2/product/get_item_base_info';
        $timestamp = time();
        $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
        $sign = hash_hmac('sha256', $baseString, $partnerKey);
        
        // item_id_list deve ser passado como query parameter separado por vírgula
        $itemIdList = implode(',', $chunk);
        $url = sprintf(
            'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s&item_id_list=%s',
            $path, $partnerId, $shopId, $timestamp, $sign, $accessToken, $itemIdList
        );
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json']
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($debug && $chunkIndex === 0) {
            $debugInfo['first_base_info_http_code'] = $httpCode;
            $debugInfo['first_base_info_response_preview'] = substr($response, 0, 500);
        }
        
        $data = json_decode($response, true);
        if (!isset($data['response']['item_list'])) {
            continue;
        }
        
        foreach ($data['response']['item_list'] as $item) {
            $debugInfo['total_items_checked']++;
            
            // Adiciona shop_id ao item para uso posterior
            $item['shop_id'] = $shopId;
            
            // Coleta amostra de SKUs para debug (primeiros 10)
            if (count($debugInfo['sample_skus']) < 10 && isset($item['item_sku'])) {
                $debugInfo['sample_skus'][] = [
                    'original' => $item['item_sku'],
                    'normalized' => strtoupper(trim(explode('#', $item['item_sku'])[0]))
                ];
            }
            
            // Verifica item_sku (SKU principal do produto)
            if (isset($item['item_sku'])) {
                $itemSku = strtoupper(trim(explode('#', $item['item_sku'])[0]));
                if ($itemSku === $skuBusca) {
                    if ($debug) {
                        $debugInfo['match_found'] = true;
                        $debugInfo['match_item_id'] = $item['item_id'];
                        logDebug('Shopee busca debug (MATCH): ' . json_encode($debugInfo));
                    }
                    return $item;
                }
            }
        }
    }
    
    if ($debug) {
        $debugInfo['match_found'] = false;
        logDebug('Shopee busca debug (NO MATCH): ' . json_encode($debugInfo));
    }
    
    return null;
}

function zerarEstoqueML($produto, $accessToken) {
    $itemId = $produto['id'];
    $status = $produto['status'] ?? 'active';
    
    // Se o produto estiver fechado (closed), apenas retorna sucesso sem fazer nada
    // Produtos closed já estão fora de venda, não precisa pausar
    if ($status === 'closed') {
        return ['success' => true, 'http_code' => 200, 'message' => 'Produto já está closed (fora de venda)'];
    }
    
    // Se estiver pausado, reativa antes de zerar
    if ($status === 'paused') {
        $urlReactivate = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
        $payloadReactivate = json_encode(['status' => 'active']);
        
        $chReactivate = curl_init($urlReactivate);
        curl_setopt_array($chReactivate, [
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_POSTFIELDS => $payloadReactivate,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ]
        ]);
        
        curl_exec($chReactivate);
        curl_close($chReactivate);
        sleep(1);
    }
    
    // Verifica se tem variações
    if (isset($produto['variations']) && is_array($produto['variations']) && count($produto['variations']) > 0) {
        // Produto com variações - zera cada variação
        $sucessos = 0;
        $erros = [];
        
        foreach ($produto['variations'] as $variation) {
            if (!isset($variation['id'])) continue;
            
            $varId = $variation['id'];
            $url = 'https://api.mercadolibre.com/items/' . urlencode($itemId) . '/variations/' . urlencode($varId);
            $payload = json_encode(['available_quantity' => 0]);
            
            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_CUSTOMREQUEST => 'PUT',
                CURLOPT_POSTFIELDS => $payload,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $accessToken
                ]
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode >= 200 && $httpCode < 300) {
                $sucessos++;
            } else {
                $responseData = json_decode($response, true);
                $errorDetail = isset($responseData['message']) ? $responseData['message'] : "HTTP {$httpCode}";
                $erros[] = "Variação {$varId}: {$errorDetail}";
            }
        }
        
        if ($sucessos > 0 && count($erros) == 0) {
            return ['success' => true, 'http_code' => 200, 'message' => "{$sucessos} variações zeradas"];
        } elseif ($sucessos > 0) {
            return ['success' => true, 'http_code' => 207, 'message' => "{$sucessos} zeradas, " . count($erros) . " erros", 'errors' => $erros];
        } else {
            // Se todas variações falharam, tenta pausar o produto inteiro
            $urlPauseFallback = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
            $payloadPauseFallback = json_encode(['status' => 'paused']);
            
            $chPauseFallback = curl_init($urlPauseFallback);
            curl_setopt_array($chPauseFallback, [
                CURLOPT_CUSTOMREQUEST => 'PUT',
                CURLOPT_POSTFIELDS => $payloadPauseFallback,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $accessToken
                ]
            ]);
            
            $responsePauseFallback = curl_exec($chPauseFallback);
            $httpCodePauseFallback = curl_getinfo($chPauseFallback, CURLINFO_HTTP_CODE);
            curl_close($chPauseFallback);
            
            if ($httpCodePauseFallback >= 200 && $httpCodePauseFallback < 300) {
                return ['success' => true, 'http_code' => $httpCodePauseFallback, 'message' => 'Produto pausado'];
            }
            
            return ['success' => false, 'error' => 'Falha ao zerar variações: ' . implode('; ', $erros)];
        }
    } else {
        // Produto simples sem variações
        $url = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
        $payload = json_encode(['available_quantity' => 0]);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($response === false) {
            return ['success' => false, 'error' => 'curl_error: ' . $error];
        }
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'http_code' => $httpCode];
        }
        
        // Se falhou ao zerar, tenta pausar como fallback
        $urlPauseFallback = 'https://api.mercadolibre.com/items/' . urlencode($itemId);
        $payloadPauseFallback = json_encode(['status' => 'paused']);
        
        $chPauseFallback = curl_init($urlPauseFallback);
        curl_setopt_array($chPauseFallback, [
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_POSTFIELDS => $payloadPauseFallback,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ]
        ]);
        
        $responsePauseFallback = curl_exec($chPauseFallback);
        $httpCodePauseFallback = curl_getinfo($chPauseFallback, CURLINFO_HTTP_CODE);
        curl_close($chPauseFallback);
        
        if ($httpCodePauseFallback >= 200 && $httpCodePauseFallback < 300) {
            return ['success' => true, 'http_code' => $httpCodePauseFallback, 'message' => 'Produto pausado'];
        }
        
        $responseData = json_decode($response, true);
        $errorMsg = 'HTTP ' . $httpCode;
        if (isset($responseData['message'])) {
            $errorMsg .= ': ' . $responseData['message'];
        } elseif (isset($responseData['error'])) {
            $errorMsg .= ': ' . $responseData['error'];
        }
        
        return ['success' => false, 'http_code' => $httpCode, 'error' => $errorMsg];
    }
}

function zerarEstoqueShopee($produto, $accessToken, $partnerId, $partnerKey, $shopId = null) {
    $itemId = $produto['item_id'];
    
    // Tenta pegar shop_id do produto ou usa o passado por parâmetro
    if (empty($shopId)) {
        $shopId = $produto['shop_id'] ?? '';
    }
    
    if (empty($shopId)) {
        return ['success' => false, 'error' => 'shop_id não encontrado'];
    }
    
    $path = '/api/v2/product/update_stock';
    $timestamp = time();
    $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
    $sign = hash_hmac('sha256', $baseString, $partnerKey);
    
    $url = sprintf(
        'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s',
        $path, $partnerId, $shopId, $timestamp, $sign, $accessToken
    );
    
    // Payload completo para produtos sem variação (mesma estrutura do v2)
    $body = [
        'item_id' => (int)$itemId,
        'stock_list' => [
            [
                'model_id' => 0,
                'stock' => 0,  // Stock no nível do model
                'seller_stock' => [
                    [
                        'location' => 'default',
                        'stock' => 0
                    ]
                ]
            ]
        ]
    ];
    
    $payload = json_encode($body);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_HTTPHEADER => ['Content-Type: application/json']
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($response === false) {
        return ['success' => false, 'error' => 'curl_error: ' . $error];
    }
    
    $data = json_decode($response, true);
    $success = false;
    
    if (isset($data['response']['success_list']) && !empty($data['response']['success_list'])) {
        $success = true;
    } elseif (isset($data['response']['failure_list']) && !empty($data['response']['failure_list'])) {
        $failureInfo = $data['response']['failure_list'][0] ?? [];
        $errorMsg = $failureInfo['failed_reason'] ?? 'Falha desconhecida';
        return ['success' => false, 'error' => $errorMsg, 'http_code' => $httpCode];
    } elseif (isset($data['error'])) {
        return ['success' => false, 'error' => $data['error'], 'http_code' => $httpCode];
    } elseif ($httpCode >= 200 && $httpCode < 300) {
        $success = true;
    }
    
    if ($success) {
        return ['success' => true, 'http_code' => $httpCode];
    }
    
    return [
        'success' => false, 
        'http_code' => $httpCode, 
        'error' => $data['message'] ?? $data['error'] ?? 'Falha desconhecida',
        'response' => $response
    ];
}

// ==================== FUNÇÕES DE RESTAURAÇÃO DE ESTOQUE ====================

function restaurarEstoqueML($loja, $produto, $debug = false) {
    global $pdo;
    
    if ($debug) {
        logDebug("=== RESTAURAR ESTOQUE MERCADO LIVRE ===");
        logDebug("Produto ID: {$produto['id']} | Estoque DB: {$produto['estoque']}");
    }
    
    $resultado = [
        'restaurado' => false,
        'produtos_restaurados' => 0,
        'error' => null
    ];
    
    $skuNormalizado = strtoupper(trim(explode('#', $produto['sku'])[0]));
    
    // Buscar todos os produtos do ML com esse SKU
    $produtosML = buscarTodosProdutosML($skuNormalizado, $loja['access_token'], $loja['seller_id'], $debug);
    
    if (empty($produtosML)) {
        $resultado['error'] = 'Produtos não encontrados para restauração';
        return $resultado;
    }
    
    $estoqueReal = (int)$produto['estoque'];
    $erros = [];
    
    foreach ($produtosML as $produtoML) {
        $itemId = $produtoML['id'];
        
        // Determina a quantidade correta baseada no status
        $status = $produtoML['status'] ?? 'active';
        
        // Se está closed, mantém em 0 (produto encerrado permanentemente)
        // Se está paused, REATIVA com estoque (era pausado só para zerar)
        if ($status === 'closed') {
            $quantidade = 0;
            logDebug("ML Restaurar {$itemId}: Produto closed, mantendo em 0");
        } else {
            // Paused ou Active: restaura com estoque real e ativa
            $quantidade = $estoqueReal;
            $status_novo = 'active'; // Força reativação
        }
        
        $urlUpdate = "https://api.mercadolibre.com/items/{$itemId}";
        
        // Monta payload: sempre atualiza estoque, e reativa se necessário
        $payload = ['available_quantity' => $quantidade];
        if (isset($status_novo)) {
            $payload['status'] = $status_novo;
        }
        
        $data = json_encode($payload);
        
        $ch = curl_init($urlUpdate);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $loja['access_token'],
                'Content-Type: application/json'
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($debug) {
            $statusInfo = isset($status_novo) ? " | Status: {$status} -> {$status_novo}" : " | Status: {$status}";
            logDebug("ML Restaurar {$itemId}: HTTP $httpCode | Qtd: $quantidade" . $statusInfo);
        }
        
        if ($httpCode >= 200 && $httpCode < 300) {
            $resultado['produtos_restaurados']++;
        } else {
            $responseData = json_decode($response, true);
            $erros[] = $itemId . ': ' . ($responseData['message'] ?? 'Erro desconhecido');
        }
    }
    
    if ($resultado['produtos_restaurados'] > 0) {
        $resultado['restaurado'] = true;
        if (!empty($erros)) {
            $resultado['error'] = 'Alguns falharam: ' . implode('; ', $erros);
        }
    } else {
        $resultado['error'] = 'Falha ao restaurar: ' . implode('; ', $erros);
    }
    
    return $resultado;
}

function restaurarEstoqueShopee($loja, $produto, $debug = false) {
    global $pdo;
    
    if ($debug) {
        logDebug("=== RESTAURAR ESTOQUE SHOPEE ===");
        logDebug("Produto SKU: {$produto['sku']} | Estoque DB: {$produto['estoque']}");
    }
    
    $resultado = [
        'restaurado' => false,
        'error' => null
    ];
    
    $skuNormalizado = strtoupper(trim(explode('#', $produto['sku'])[0]));
    $shopId = $loja['seller_id'];
    
    // Buscar produto na Shopee
    $produtoShopee = buscarProdutoShopee(
        $skuNormalizado,
        $loja['access_token'],
        $shopId,
        $loja['client_id'],
        $loja['client_secret'],
        $debug
    );
    
    if (!$produtoShopee) {
        $resultado['error'] = 'Produto não encontrado para restauração';
        return $resultado;
    }
    
    $itemId = $produtoShopee['item_id'];
    $estoqueReal = (int)$produto['estoque'];
    
    if ($debug) {
        logDebug("Shopee Restaurar - Item ID: {$itemId} | Estoque: {$estoqueReal}");
    }
    
    // Restaurar estoque na Shopee
    $path = '/api/v2/product/update_stock';
    $timestamp = time();
    $accessToken = $loja['access_token'];
    $partnerId = $loja['client_id'];
    $partnerKey = $loja['client_secret'];
    
    $baseString = $partnerId . $path . $timestamp . $accessToken . $shopId;
    $sign = hash_hmac('sha256', $baseString, $partnerKey);
    
    $url = sprintf(
        'https://partner.shopeemobile.com%s?partner_id=%s&shop_id=%s&timestamp=%s&sign=%s&access_token=%s',
        $path, $partnerId, $shopId, $timestamp, $sign, $accessToken
    );
    
    // Shopee exige seller_stock com location para produtos sem variação
    $stockData = [
        'item_id' => $itemId,
        'stock_list' => [
            [
                'model_id' => 0,
                'stock' => $estoqueReal,  // Stock no nível do model
                'seller_stock' => [
                    [
                        'location' => 'default',
                        'stock' => $estoqueReal
                    ]
                ]
            ]
        ]
    ];
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($stockData),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json']
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($debug) {
        logDebug("Shopee Restaurar Item {$itemId}: HTTP $httpCode | Qtd: $estoqueReal");
        logDebug("Response: $response");
    }
    
    $data = json_decode($response, true);
    
    if ($httpCode >= 200 && $httpCode < 300) {
        if (isset($data['response']['success_list']) && !empty($data['response']['success_list'])) {
            $resultado['restaurado'] = true;
        } elseif (isset($data['response']['failure_list']) && !empty($data['response']['failure_list'])) {
            $failureInfo = $data['response']['failure_list'][0] ?? [];
            $resultado['error'] = $failureInfo['failed_reason'] ?? 'Falha desconhecida';
        } else {
            $resultado['restaurado'] = true; // Assume sucesso se não há erro explícito
        }
    } else {
        $resultado['error'] = $data['message'] ?? $data['error'] ?? 'Erro HTTP ' . $httpCode;
    }
    
    return $resultado;
}

