<?php
session_start();
date_default_timezone_set('America/Sao_Paulo');

require '../../class/configClass.php';
require '../../vendor/autoload.php';
Config::load('../../config/config.php');
include_once('../../conexao/conexao.php');


// Valida entrada
if (!isset($_POST['saleTitles']) || empty(trim($_POST['saleTitles']))) {
    echo json_encode([
        'success' => false,
        'message' => 'Nenhum código de venda foi informado.'
    ]);
    exit();
}

// Parse dos códigos de venda
$saleTitlesRaw = $_POST['saleTitles'];
$saleTitlesArray = preg_split('/[\s,;]+/', $saleTitlesRaw, -1, PREG_SPLIT_NO_EMPTY);
$saleTitlesArray = array_unique(array_map('trim', $saleTitlesArray));

if (empty($saleTitlesArray)) {
    echo json_encode([
        'success' => false,
        'message' => 'Nenhum código de venda válido foi informado.'
    ]);
    exit();
}

// Configurações da API
$apiBaseUrl = Config::get('api_configs.base_url');
$apiToken = Config::get('api_configs.token');

if (empty($apiBaseUrl) || empty($apiToken)) {
    echo json_encode([
        'success' => false,
        'message' => 'Configurações da API não encontradas.'
    ]);
    exit();
}

// Monta os placeholders e params para o IN clause
$placeholders = [];
$params = [];
foreach ($saleTitlesArray as $index => $saleTitle) {
    $placeholder = ":id{$index}";
    $placeholders[] = $placeholder;
    $params[$placeholder] = $saleTitle;
}

// Monta a SQL com IN clause
$sql = "SELECT 
    saleTitle, 
    fornecedor, 
    mv.usuario_id as id_usuario, 
    whitelabel, 
    LOWER(mv.integracao) as plataforma, 
    w.url_base 
FROM drophub_global.minhas_vendas mv 
INNER JOIN drophub_global.whitelabels w ON mv.whitelabel = w.nome 
WHERE mv.saleTitle IN (" . implode(', ', $placeholders) . ")";

// Prepara o payload
$payload = json_encode([
    'sql' => $sql,
    'params' => $params
]);

// Executa a consulta via API
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => rtrim($apiBaseUrl, '/') . '/query/custom',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_ENCODING => '',
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => 'POST',
    CURLOPT_POSTFIELDS => $payload,
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiToken
    ],
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curlError = curl_error($ch);
curl_close($ch);

if ($curlError) {
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao consultar API: ' . $curlError
    ]);
    exit();
}

if ($httpCode !== 200) {
    echo json_encode([
        'success' => false,
        'message' => 'API retornou erro HTTP ' . $httpCode
    ]);
    exit();
}

$apiResponse = json_decode($response, true);

if (!$apiResponse || !isset($apiResponse['success']) || !$apiResponse['success']) {
    echo json_encode([
        'success' => false,
        'message' => 'Erro na resposta da API: ' . ($apiResponse['message'] ?? 'Resposta inválida')
    ]);
    exit();
}

// Processa as vendas encontradas
$vendasEncontradas = $apiResponse['data']['data'] ?? [];
$vendasEncontradasCodes = array_column($vendasEncontradas, 'saleTitle');
$vendasNaoEncontradas = array_diff($saleTitlesArray, $vendasEncontradasCodes);

$jobsCriados = 0;
$jobsResetados = 0;
$jobsEmProcessamento = 0;
$erros = 0;
$detalhes = [];
$errosLista = [];

// Cria os jobs para cada venda encontrada
foreach ($vendasEncontradas as $venda) {
    try {
        // Verificar se já existe job para esta venda
        $checkStmt = $pdo->prepare("
            SELECT id, status, tentativas, max_tentativas, erro_mensagem
            FROM drophub_global.jobs_etiquetas 
            WHERE sale_title = :sale_title 
            AND fornecedor = :fornecedor
            AND whitelabel = :whitelabel
            ORDER BY created_at DESC 
            LIMIT 1
        ");
        
        $checkStmt->execute([
            ':sale_title' => $venda['saleTitle'],
            ':fornecedor' => $venda['fornecedor'],
            ':whitelabel' => $venda['whitelabel']
        ]);
        
        $existingJob = $checkStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingJob) {
            // Se job está sendo processado, não resetar
            if ($existingJob['status'] === 'processing') {
                $jobsEmProcessamento++;
                $venda['job_id'] = $existingJob['id'];
                $venda['job_status'] = 'processing';
                $venda['acao'] = 'em_processamento';
                $detalhes[] = $venda;
                
                error_log("⏳ Reprocessamento - Job já está sendo processado - ID: {$existingJob['id']} | Venda: {$venda['saleTitle']} | Fornecedor: {$venda['fornecedor']}");
                continue;
            }
            
            // Para outros status, resetar para pending
            $updateStmt = $pdo->prepare("
                UPDATE drophub_global.jobs_etiquetas 
                SET status = 'pending', 
                    tentativas = 0,
                    erro_mensagem = NULL,
                    url = :url,
                    updated_at = NOW()
                WHERE id = :id
            ");
            
            $updateResult = $updateStmt->execute([
                ':id' => $existingJob['id'],
                ':url' => $venda['url_base']
            ]);
            
            if ($updateResult) {
                $jobsResetados++;
                $venda['job_id'] = $existingJob['id'];
                $venda['job_status'] = 'pending';
                $venda['acao'] = 'resetado';
                $venda['status_anterior'] = $existingJob['status'];
                $detalhes[] = $venda;
                
                error_log("🔄 Reprocessamento MANUAL - Job resetado - ID: {$existingJob['id']} | Status anterior: {$existingJob['status']} | Venda: {$venda['saleTitle']} | Fornecedor: {$venda['fornecedor']}");
            } else {
                $erros++;
                $errosLista[] = "Erro ao resetar job para venda {$venda['saleTitle']}";
                error_log("❌ Reprocessamento - Erro ao resetar job - ID: {$existingJob['id']} | Venda: {$venda['saleTitle']}");
            }
            continue;
        }
        
        // Se não existe job, criar novo
        $insertStmt = $pdo->prepare("
            INSERT INTO drophub_global.jobs_etiquetas (
                sale_title,
                fornecedor,
                id_usuario,
                whitelabel,
                plataforma,
                url,
                status,
                tentativas
            ) VALUES (?, ?, ?, ?, ?, ?, 'pending', 0)
        ");
        
        $result = $insertStmt->execute([
            $venda['saleTitle'],
            $venda['fornecedor'],
            $venda['id_usuario'],
            $venda['whitelabel'],
            $venda['plataforma'],
            $venda['url_base']
        ]);
        
        if ($result) {
            $jobId = $pdo->lastInsertId();
            $jobsCriados++;
            
            $venda['job_id'] = $jobId;
            $venda['job_status'] = 'pending';
            $venda['acao'] = 'criado';
            $detalhes[] = $venda;
            
            error_log("✅ Reprocessamento - Job novo criado - ID: {$jobId} | Venda: {$venda['saleTitle']} | Fornecedor: {$venda['fornecedor']} | Plataforma: {$venda['plataforma']} | Whitelabel: {$venda['whitelabel']}");
        } else {
            $erros++;
            $errosLista[] = "Erro ao criar job para venda {$venda['saleTitle']}";
            error_log("❌ Reprocessamento - Erro ao criar job para venda: {$venda['saleTitle']}");
        }
    } catch (Exception $e) {
        $erros++;
        $errosLista[] = "Exceção ao processar venda {$venda['saleTitle']}: " . $e->getMessage();
        error_log("❌ Reprocessamento - Exceção ao processar: " . $e->getMessage());
    }
}

// Resposta final
echo json_encode([
    'success' => true,
    'message' => 'Processamento concluído',
    'total_solicitadas' => count($saleTitlesArray),
    'total_processadas' => count($vendasEncontradas),
    'jobs_criados' => $jobsCriados,
    'jobs_resetados' => $jobsResetados,
    'jobs_em_processamento' => $jobsEmProcessamento,
    'nao_encontradas' => count($vendasNaoEncontradas),
    'nao_encontradas_lista' => array_values($vendasNaoEncontradas),
    'erros' => $erros,
    'erros_lista' => $errosLista,
    'detalhes' => $detalhes
]);
