<?php
session_start();
date_default_timezone_set('America/Sao_Paulo');

header('Content-Type: application/json');

require '../../class/configClass.php';
Config::load('../../config/config.php');
include_once('../../conexao/conexao.php');

// Verificar se está logado
if (!isset($_SESSION['usuario']['email'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autorizado']);
    exit();
}


try {
    // Receber IDs dos jobs
    $jobIds = $_POST['job_ids'] ?? [];
    
    if (empty($jobIds) || !is_array($jobIds)) {
        echo json_encode(['success' => false, 'message' => 'IDs de jobs não fornecidos']);
        exit();
    }
    
    // Preparar placeholders para a query
    $placeholders = implode(',', array_fill(0, count($jobIds), '?'));
    
    // Buscar status dos jobs
    $sql = "
        SELECT 
            id,
            sale_title,
            fornecedor,
            status,
            tentativas,
            max_tentativas,
            erro_mensagem,
            created_at,
            processed_at,
            updated_at,
            url as etiqueta_url
        FROM drophub_global.jobs_etiquetas
        WHERE id IN ($placeholders)
        ORDER BY id ASC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($jobIds);
    $jobs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular estatísticas
    $stats = [
        'total' => count($jobs),
        'pending' => 0,
        'processing' => 0,
        'completed' => 0,
        'failed' => 0
    ];
    
    $jobsDetalhados = [];
    
    foreach ($jobs as $job) {
        $stats[$job['status']]++;
        
        // Formatar dados do job
        $jobsDetalhados[] = [
            'id' => $job['id'],
            'sale_title' => $job['sale_title'],
            'fornecedor' => $job['fornecedor'],
            'status' => $job['status'],
            'tentativas' => $job['tentativas'],
            'max_tentativas' => $job['max_tentativas'],
            'erro_mensagem' => $job['erro_mensagem'],
            'etiqueta_url' => $job['etiqueta_url'],
            'created_at' => $job['created_at'],
            'processed_at' => $job['processed_at'],
            'updated_at' => $job['updated_at'],
            'tempo_decorrido' => calcularTempoDecorrido($job['created_at'])
        ];
    }
    
    // Verificar se todos os jobs foram finalizados
    $todosFinalizados = ($stats['pending'] + $stats['processing']) === 0;
    
    echo json_encode([
        'success' => true,
        'jobs' => $jobsDetalhados,
        'stats' => $stats,
        'todos_finalizados' => $todosFinalizados,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    error_log("Erro ao consultar status de jobs: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao consultar status dos jobs: ' . $e->getMessage()
    ]);
}

/**
 * Calcula tempo decorrido desde a criação
 */
function calcularTempoDecorrido($dataInicio) {
    $inicio = new DateTime($dataInicio);
    $agora = new DateTime();
    $intervalo = $inicio->diff($agora);
    
    if ($intervalo->h > 0) {
        return $intervalo->h . 'h ' . $intervalo->i . 'min';
    } elseif ($intervalo->i > 0) {
        return $intervalo->i . 'min ' . $intervalo->s . 's';
    } else {
        return $intervalo->s . 's';
    }
}
